const HighchartsFormatter = (function () {

    const self = {};

    function formatPoints (points, group) {
        return points.map(point => {
            var point = { x: point.instant, y: point.value, group: group };
            return point;
        });
    }

    function absoluteValue(point, decimalPlaces) {
        var decimalPlaces = decimalPlaces(point.series.name);
        return decimalPlaces != null ? point.y.toFixed(decimalPlaces.absolute) : 0;
    }

    function percentValue(point, total, decimalPlaces) {
        if (total == 0) return 0;
        return (point.y * 100 / total).toFixed(decimalPlaces(point.series.name).percentage);
    }

    function tooltipLabel(points, point, metric, decimalPlaces, shouldDisplayPercentage) {
        if (point.series.name == '_eventsSeries') return '';
        const symbolAndSeriesName = `<span style="color: ${point.series.color}; font-size: 14pt">\u25CF</span>${point.series.name}`;
        let value = `${absoluteValue(point, decimalPlaces)}`;
        if (shouldDisplayPercentage(point.series.name)) {
            value += ` (${percentValue(point, groupTotal(points, point), decimalPlaces)} %)`;
        }
        return `${symbolAndSeriesName}: <b>${value} ${metric(point.series.name)}</b><br/>`;
    }

    function groupTotal(points, p) {
        var groupForPoint = pointGroup(findPoint(points, p.x, p.y));
        return points.filter(p => p.point.group == groupForPoint).map(point => point.y).reduce((l, r) => l + r);
    }

    function findPoint(points, x, y) {
        return points.filter(point => point.x == x && point.y == y)[0];
    }

    function pointGroup(p) {
        return p.point.group;
    }

    self.stockChart = function (options) {
        return {
            chart: {
                renderTo: options.container
            },

            navigator: {
                enabled: false
            },

            credits: {
                enabled: false
            },

            scrollbar : {
                enabled : false
            },

            rangeSelector: {
                enabled: false
            },

            xAxis: {
                labels: {
                    enabled: false
                },
                minorTickLength: 0,
                tickLength: 0
            },

            yAxis: {
                gridLineDashStyle: 'Dot',
                labels: {
                    style: {
                        fontSize: '8pt',
                        fontFamily: 'Rubik, Helvetica Neue, Helvetica, Arial, sans-serif'
                    }
                }
            },

            tooltip: {
                formatter: function() {
                    if (!this.points) {
                        return !this.point.text ? false : this.point.text;
                    }
                    return this.points
                    .map(point => tooltipLabel(this.points, point, options.metric, options.decimalPlaces, options.shouldDisplayPercentage))
                    .reduce((accumulated, newValue) => accumulated += newValue, `<b>${options.formatInstant(new Date(this.x))}</b><br/>`);
                }
            },

            plotOptions: {
                series: {
                    point: {
                        events: {
                            mouseOver: function () {
                                options.onMouseOver(this.x);
                            }
                        }
                    }
                }
            }
        };
    };

    self.series = function (options) {
        if (options.indicator.style === 'absolute' || options.indicator.style === 'percentage') {
            return {
                name: options.indicator.label,
                type: 'area',
                data: formatPoints(options.indicator.dataList, options.indicator.group),
                color: options.indicator.color,
                fillColor: options.indicator.color,
                threshold: null,
                stacking: options.indicator.style === 'absolute' ? 'normal' : 'percent',
                yAxis: options.axis,
                zIndex: 1,
                marker : {
                    enabled : true,
                    radius : 2
                }
            };
        }
        return {
            name: options.indicator.label,
            data: formatPoints(options.indicator.dataList, options.indicator.group),
            color: options.indicator.color,
            yAxis: options.axis,
            lineWidth: options.indicator.style === 'dotted' ? 0 : 1,
            zIndex: 2,
            marker : {
                enabled : true,
                radius : 2
            }
        };
    };

    self.addPoints = function (series, group, points) {
        const instantsAdded = series.points != null ? series.points.map(point => point.x) : [];
        formatPoints(points.filter(point => instantsAdded.indexOf(point.instant) < 0), group).forEach(point => series.addPoint(point, false, false, false));
    };

    return self;
})();

window.HighchartsFormatter = HighchartsFormatter;
