package io.intino.sumus.reporting.loaders;

import io.intino.sumus.reporting.Node;

import java.util.*;
import java.util.stream.Collectors;

public class NodeLoader {

	private final Map<String, Node> nodeMap = new HashMap<>();
	private final String dimension;
	private final Set<String> names;

	public NodeLoader(String dimension, Set<String> names) {
		this.dimension = dimension;
		this.names = names;
	}

	public List<Node> nodes() {
		for (String name : names) getOrCreate(name);
		List<Node> nodes = new ArrayList<>(nodeMap.values());
		nodes.add(globalNode(nodes));
		return nodes;
	}

	private Node getOrCreate(String name) {
		if (!nodeMap.containsKey(name) && names.contains(name)) nodeMap.put(name, nodeOf(name));
		return nodeMap.get(name);
	}

	private Node nodeOf(String name) {
		return new Node() {

			@Override
			public String id() {
				return name;
			}

			@Override
			public String name() {
				return name;
			}

			@Override
			public String dimension() {
				return dimension;
			}

			@Override
			public Node parent() {
				if (!name.contains(".")) return null;
				return getOrCreate(name.substring(0, name.lastIndexOf('.')));
			}

			@Override
			public List<Node> children() {
				return names.stream()
						.filter(o -> o.startsWith(name + "."))
						.filter(o -> !(o.replace(name + ".", "").contains(".")))
						.map(n-> getOrCreate(n))
						.collect(Collectors.toList());
			}
		};
	}

	private Node globalNode(List<Node> nodes) {
		return new Node() {

			@Override
			public String id() {
				return GlobalNode;
			}

			@Override
			public String name() {
				return "";
			}

			@Override
			public String dimension() {
				return dimension;
			}

			@Override
			public Node parent() {
				return null;
			}

			@Override
			public List<Node> children() {
				return nodes.stream().filter(n -> n.parent() == null).filter(n -> !n.id().equals(GlobalNode)).collect(Collectors.toList());
			}
		};
	}
}
