package io.intino.sumus.reporting.loaders;

import io.intino.alexandria.Timetag;
import io.intino.alexandria.logger.Logger;
import io.intino.sumus.engine.Ledger;
import io.intino.sumus.engine.LedgerDecorator;
import io.intino.sumus.engine.ledgers.EmptyLedger;
import io.intino.sumus.engine.ledgers.columnar.ColumnarLedger;
import io.intino.sumus.engine.ledgers.columnar.ColumnarLedgerDecorator;
import io.intino.sumus.engine.ledgers.composite.CompositeLedger;
import io.intino.sumus.model.LedgerDefinition;
import io.intino.sumus.reporting.model.Period;
import io.intino.sumus.reporting.model.Scale;

import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.util.HashMap;
import java.util.Map;

import static io.intino.sumus.reporting.loaders.LedgerAggregator.isAggregable;

public class LedgerLoader {

	private final File folder;
	private final LedgerDecorator<ColumnarLedger> decorator;
	private final LedgerAggregator aggregator;

	public LedgerLoader(File folder) {
		this.folder = folder;
		this.decorator = new ColumnarLedgerDecorator(folder);
		this.aggregator = new LedgerAggregator(folder);
	}

	public synchronized Ledger ledger(String name, Scale scale, Timetag timetag) {
		return ledger(name, new Period(scale), timetag);
	}

	public synchronized Ledger ledger(String name, Period period, Timetag timetag) {
		return ledger(name, period.timetags(timetag).toArray(Timetag[]::new));
	}

	public synchronized Ledger ledger(String name, Timetag... timetags) {
		try {
			return loadLedger(name, timetags);
		} catch (LedgerNotFoundException e) {
			return new EmptyLedger();
		} catch (Throwable e) {
			Logger.warn("Error loading ledger: " +  name);
			return new EmptyLedger();
		}
	}

	private Ledger loadLedger(String name, Timetag... timetags) throws IOException, LedgerNotFoundException, ParseException {
		Map<Timetag, Ledger> ledgers = loadLedgers(name, timetags);
		if (ledgers.isEmpty()) throw new LedgerNotFoundException();
		if (ledgers.size() == 1) return ledgers.values().stream().findFirst().orElseThrow(LedgerNotFoundException::new);

		CompositeLedger composite = new CompositeLedger("date");
		ledgers.forEach((timetag, ledger) -> composite.add(ledger, timetag));
		return composite;
	}

	private Map<Timetag, Ledger> loadLedgers(String name, Timetag... timetags) throws IOException, ParseException {
		LedgerDefinition definition = definition(name);
		Map<Timetag, Ledger> ledgers = new HashMap<>();
		for (Timetag timtag : timetags) {
			Ledger ledger = loadLedger(name, definition, timtag);
			if (ledger != null) ledgers.put(timtag, ledger);
		}
		return ledgers;
	}

	private Ledger loadLedger(String name, LedgerDefinition definition, Timetag timetag) throws IOException {
		File file = ledgerFile(name, timetag);
		if (file.exists()) return ledger(definition, file);
		if (isAggregable(definition)) return aggregatedLedger(name, definition, timetag);
		return null;
	}

	private Ledger aggregatedLedger(String name, LedgerDefinition definition, Timetag timetag) {
		try {
			String[][] content = aggregator.aggregate(name, definition, timetag);
			return ledger(definition, content);
		} catch (Throwable e) {
			Logger.error(e);
			return null;
		}
	}

	private Ledger ledger(LedgerDefinition definition, File file) throws IOException {
		return decorator.decorate(new ColumnarLedger(definition).load(file, "\t"));
	}

	private Ledger ledger(LedgerDefinition definition, String[][] content) {
		return decorator.decorate(new ColumnarLedger(definition).load(content));
	}

	private LedgerDefinition definition(String type) throws ParseException {
		return LedgerDefinition.load(folder, new File(folder, type + ".ledger"));
	}

	private File ledgerFile(String type, Timetag timetag) {
		return new File(folder, type + "/" + timetag + ".tsv");
	}

	public static class LedgerNotFoundException extends Exception {
	}

//	public synchronized Ledger ledger(String name, Period period, LocalDate date) {
//		return ledger(name, period.dates(date).toArray(LocalDate[]::new));
//	}
//
//	public synchronized Ledger ledger(String name, LocalDate... dates) {
//		try {
//			return loadLedger(name, dates);
//		} catch (LedgerNotFoundException e) {
//			return new EmptyLedger();
//		} catch (Throwable e) {
//			Logger.warn("Error loading ledger: " +  name);
//			return new EmptyLedger();
//		}
//	}
//
//	private Ledger loadLedger(String name, LocalDate... dates) throws IOException, LedgerNotFoundException, ParseException {
//		Map<LocalDate, Ledger> ledgers = loadLedgers(name, dates);
//		if (ledgers.isEmpty()) throw new LedgerNotFoundException();
//		if (ledgers.size() == 1) return ledgers.values().stream().findFirst().orElseThrow(LedgerNotFoundException::new);
//
//		CompositeLedger composite = new CompositeLedger("date");
//		ledgers.forEach((date, ledger) -> composite.add(ledger, date));
//		return composite;
//	}
//
//	private Map<LocalDate, Ledger> loadLedgers(String name, LocalDate... dates) throws IOException, ParseException {
//		LedgerDefinition definition = definition(name);
//		Map<LocalDate, Ledger> ledgers = new HashMap<>();
//		for (LocalDate date : dates) {
//			Ledger ledger = loadLedger(name, definition, date);
//			if (ledger != null) ledgers.put(date, ledger);
//		}
//		return ledgers;
//	}
//
//	private Ledger loadLedger(String name, LedgerDefinition definition, LocalDate date) throws IOException {
//		File file = ledgerFile(name, date);
//		if (file.exists()) return ledger(definition, file);
//		if (isAggregable(definition)) return aggregatedLedger(name, definition, date);
//		return null;
//	}
//
//	private Ledger aggregatedLedger(String name, LedgerDefinition definition, LocalDate date) {
//		return aggregatedLedger(name, definition, new Timetag(date, Scale.Day));
//	}
//
//	private File ledgerFile(String type, LocalDate date) {
//		return new File(folder, type + "/" + timetag(date) + ".tsv");
//	}
}
