package io.intino.sumus.reporting.calendars;

import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.temporal.IsoFields;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.WeekFields;
import java.util.List;
import java.util.stream.Collectors;

public class StandardWeekCalendar implements WeekCalendar {

	public final DayOfWeek WeekStart;

	public StandardWeekCalendar() {
		this(DayOfWeek.MONDAY);
	}

	public StandardWeekCalendar(DayOfWeek weekStart) {
		WeekStart = weekStart;
	}

	@Override
	public List<LocalDate> dates(LocalDate to) {
		return startDate(to).datesUntil(to.plusDays(1)).collect(Collectors.toList());
	}

	@Override
	public LocalDate startDate(LocalDate date) {
		return date.with(TemporalAdjusters.previousOrSame(WeekStart));
	}

	@Override
	public LocalDate endDate(LocalDate date) {
		return date.with(TemporalAdjusters.nextOrSame(WeekStart.minus(1)));
	}

	@Override
	public String format(LocalDate date) {
		return weekyear(date) + "-" + weekNumber(date);
	}

	@Override
	public String timetag(LocalDate date) {
		return weekyear(date) + "W" + String.format("%02d", weekNumber(date));
	}

	@Override
	public String label(LocalDate date) {
		return "W" + weekNumber(date);
	}

	private int weekNumber(LocalDate date) {
		return date.with(WeekFields.of(WeekStart, 1).dayOfWeek(), 1).get(IsoFields.WEEK_OF_WEEK_BASED_YEAR);
	}

	private int weekyear(LocalDate date) {
		return date.with(WeekFields.of(WeekStart, 1).dayOfWeek(), 1).get(IsoFields.WEEK_BASED_YEAR);
	}
}
