package io.intino.sumus.reporting.builders.schemas;

import io.intino.sumus.reporting.model.Order;

import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static io.intino.sumus.reporting.model.Order.Type.*;

public class ColumnChart {

    private final String id;
    private final String title;
    private final Order order;
    private List<Category> categories = new LinkedList<>();
    private List<Serie> series = new LinkedList<>();
    private List<PlotLine> plotLines = new LinkedList<>();
    private List<CategoryAction> onClick = new LinkedList<>();
    private String units = "";
    private boolean isStacked = false;
    private boolean showDataLabels = false;

    public ColumnChart(String id, String title) {
        this(id, title, null);
    }

    public ColumnChart(String id, String title, Order order) {
        this.id = id;
        this.title = title;
        this.order = order;
    }

    public ColumnChart units(String units) {
        this.units = units;
        return this;
    }

    public ColumnChart stacked(boolean isStacked) {
        this.isStacked = isStacked;
        return this;
    }

    public ColumnChart series(List<Serie> series) {
        this.categories = categoriesOf(series).stream().map(Category::new).collect(Collectors.toList());
        this.series = sortValues(series);
        return this;
    }

    public ColumnChart plotLines(List<PlotLine> plotLines) {
        this.plotLines = plotLines;
        return this;
    }

    public ColumnChart onClick(List<CategoryAction> onClick) {
        this.onClick = onClick;
        return this;
    }

    public ColumnChart showDataLabels(boolean show) {
        this.showDataLabels = show;
        return this;
    }

    private List<String> categoriesOf(List<Serie> series) {
        List<String> categories = allCategories(series);
        if (order == null) return categories;
        if (order.is(Alphabetically)) return categories.stream().sorted().collect(Collectors.toList());
        if (order.is(Ascending) || order.is(Descending)) return categoriesSortedNumerically(categories, series);
        return categories;
    }

    private List<String> categoriesSortedNumerically(List<String> categories, List<Serie> series) {
        Serie serie = serie(series, order.indicator());
        if (serie == null) return categories;
        return categories.stream().sorted(sortMethodOf(order, serie)).collect(Collectors.toList());
    }

    private static Comparator<String> sortMethodOf(Order order, Serie serie) {
        return (c1, c2) -> {
            Double v1 = serie.doubleValue(c1);
            Double v2 = serie.doubleValue(c2);
            return order.is(Ascending) ? v1.compareTo(v2) : v2.compareTo(v1);
        };
    }

    private List<String> allCategories(List<Serie> series) {
        return series.stream().flatMap(Serie::categoriesStream).distinct().collect(Collectors.toList());
    }

    private Serie serie(List<Serie> series, String indicator) {
        return series.stream().filter(s -> s.name().equalsIgnoreCase(indicator)).findFirst().orElse(null);
    }

    private List<Serie> sortValues(List<Serie> series) {
        return series.stream().map(s -> s.sortValuesBy(this.categories)).collect(Collectors.toList());
    }

    private static class Category {
        private final String name;

        public Category(String name) {
            this.name = name;
        }
    }

    public static class Serie {

        private final String name;
        private List<Value> values = new LinkedList<>();
        private String seriesOnClick;

        public Serie(String name) {
            this.name = name;
        }

        public Serie onClick(String action) {
            this.seriesOnClick = action;
            return this;
        }

        public String name() {
            return name;
        }

        public Stream<String> categoriesStream() {
            return values.stream().map(v -> v.category);
        }

        public boolean isEmpty() {
            return values.stream().allMatch(v -> v.value == 0);
        }

        public Double doubleValue(String category) {
            Value value = value(category);
            return value != null ? value.value : 0.0;
        }

        private Value value(String category) {
            return values.stream()
                    .filter(v -> v.category.equalsIgnoreCase(category))
                    .findFirst().orElse(null);
        }

        public Serie addValue(String category, Double value) {
            values.add(new Value(category, value));
            return this;
        }

        private Serie sortValuesBy(List<Category> categories) {
            List<Value> newValues = categories.stream()
                    .map(c -> c.name)
                    .map(c -> new Value(c, doubleValue(c)))
                    .collect(Collectors.toList());
            this.values = newValues;
            return this;
        }

        private class Value {

            private final Double value;
            private final String serie;
            private final String category;
            private String onClick;

            public Value(String category, Double value) {
                this.value = value;
                this.category = category;
                this.serie = name.toLowerCase().replace(" ", "");
                this.onClick = seriesOnClick;
            }
        }
    }

    public static class PlotLine {
        private final Double value;
        private String label;
        private String color;

        public PlotLine(Double value) {
            this.value = value;
        }

        public PlotLine(Double value, String label, String color) {
            this.value = value;
            this.label = label;
            this.color = color;
        }
    }

    public static class CategoryAction {

        private final String name;
        private final String action;

        public CategoryAction(String name, String action) {
            this.name = name;
            this.action = action;
        }
    }
}
