package io.intino.sumus.reporting.builders;

import io.intino.alexandria.Timetag;
import io.intino.sumus.reporting.Dashboard;
import io.intino.sumus.reporting.Node;
import io.intino.sumus.reporting.builders.schemas.views.ViewChart.Function;
import io.intino.sumus.reporting.builders.schemas.views.ViewChart.ViewDate;
import io.intino.sumus.reporting.builders.schemas.views.ViewConfig;
import io.intino.sumus.reporting.builders.schemas.views.ViewConfig.ViewLabels;
import io.intino.sumus.reporting.builders.schemas.views.ViewConfig.ViewPlotLine;
import io.intino.sumus.reporting.insights.TimelineInsight;
import io.intino.sumus.reporting.model.Period;
import io.intino.sumus.reporting.model.Scale;

import java.time.LocalDate;
import java.util.*;
import java.util.stream.Collectors;

import static io.intino.sumus.reporting.Dashboard.Insight;

public abstract class ViewBuilder implements UIBuilder {

    protected final TimelineInsight insight;
    protected final Timetag timetag;
    protected final String report;
    protected String chartID;

    public ViewBuilder(Dashboard.Report report, Insight insight, Timetag timetag) {
        this.insight = new TimelineInsight(insight);
        this.timetag = timetag;
        this.report = report.name();
        this.chartID = insight.id() + "-chart";
    }

    protected Function functionOf(String functionCall, Node node) {
        return new Function(functionCall, chartID, insight.label())
                .config(config(node))
                .dates(dates());
    }

    private ViewConfig config(Node node) {
        return new ViewConfig(report, insight.ledger(), insight.view())
                .indicators(List.of(insight.indicators()))
                .dimension(insight.drillDimension())
                .dimension2(insight.filterDimension())
                .nodeDimension(node.dimension())
                .node(node.name())
                .nodeID(node.id())
                .filters(List.of(insight.filters(timetag)))
                .slices(List.of(insight.slices()))
                .viewFilters(List.of(insight.viewFilter()))
                .dateFilters(List.of(insight.dateFilters(timetag)))
                .plotLines(plotLines())
                .labels(labels())
                .isNav(isNavigable(node))
                .showDataLabels(insight.showDataLabels());
    }

    private List<ViewDate> dates() {
        Period period = insight.period();
        Period timeUnit = insight.timeUnit();
        return dates(period, timeUnit).stream()
                .map(date -> viewDateOf(timeUnit, date))
                .collect(Collectors.toList());
    }

    private List<LocalDate> dates(Period period, Period timeUnit) {
        Set<String> cache = new HashSet<>();
        List<LocalDate> uniques = new LinkedList<>();
        List<LocalDate> dates = period.isSingle() ? naturalDates(period) : period.dates(timetag);
        for (LocalDate date : dates) {
            String timetag = timeUnit.scale().timetag(date);
            if (cache.contains(timetag)) continue;
            cache.add(timetag);
            uniques.add(date);
        }
        return uniques;
    }

    private static ViewDate viewDateOf(Period timeUnit, LocalDate date) {
        Scale timeUnitScale = timeUnit.scale();
        return new ViewDate(
                timeUnitScale.timetag(date),
                timeUnitScale.label(date),
                timeUnitScale.format(date),
                Scale.Day.timetag(timeUnit.onlyLast() ? timeUnitScale.endDate(date) : timeUnitScale.startDate(date)),
                Scale.Day.timetag(timeUnitScale.endDate(date))
        );
    }

    private List<LocalDate> naturalDates(Period period) {
        Scale scale = period.scale();
        LocalDate start = scale.startDate(timetag);
        LocalDate end = scale.endDate(timetag);
        return start.datesUntil(end.plusDays(1)).collect(Collectors.toList());
    }

    private boolean isNavigable(Node node) {
        return node.dimension() != null &&
                insight.navFilterEnabled() &&
                insight.dimensions().length == 1 &&
                insight.dimensions()[0].equalsIgnoreCase(node.dimension());
    }

    private List<ViewPlotLine> plotLines() {
        if (insight.plotLines() == null) return Collections.emptyList();
        return Arrays.stream(insight.plotLines().split(","))
                .map(line -> line.split(":"))
                .filter(d -> d.length >= 1)
                .map(this::plotLine)
                .collect(Collectors.toList());
    }

    private ViewPlotLine plotLine(String[] data) {
        return new ViewPlotLine(data[0])
                .name(data.length > 1 ? data[1] : null)
                .color(data.length > 2 ? data[2] : null);
    }

    private ViewLabels labels() {
        ViewLabels labels = new ViewLabels();
        insight.labels().forEach(labels::set);
        return labels;
    }
}
