package io.intino.sumus.reporting.builders;

import io.intino.alexandria.logger.Logger;
import io.intino.sumus.engine.Cube;
import io.intino.sumus.engine.Cube.Cell;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.reporting.Dashboard;
import io.intino.sumus.reporting.Dashboard.Insight;
import io.intino.sumus.reporting.Node;
import io.intino.sumus.reporting.builders.schemas.Table;
import io.intino.sumus.reporting.builders.schemas.Table.Attribute;
import io.intino.sumus.reporting.builders.schemas.Table.Row;
import io.intino.sumus.reporting.builders.schemas.Table.TableChartException;
import io.intino.sumus.reporting.builders.templates.Renderer;
import io.intino.sumus.reporting.helpers.CubesHelper;
import io.intino.sumus.reporting.insights.TableInsight;
import io.intino.sumus.reporting.model.Formatter;

import java.time.LocalDate;
import java.time.LocalTime;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static io.intino.sumus.reporting.DashboardBuilder.MaxLevel;
import static io.intino.sumus.reporting.builders.schemas.Table.sortMethodOf;
import static java.util.Arrays.stream;

public class TableBuilder implements UIBuilder {

    protected final TableInsight insight;
    private final Dashboard.Report report;
    private final MicrositeActionBuilder microsite;

    public TableBuilder(Dashboard.Report report, Insight insight, MicrositeActionBuilder microsite) {
        this.insight = new TableInsight(insight);
        this.report = report;
        this.microsite = microsite;
    }

    @Override
    public String build(Cube cube, Node node) {
        try {
            Table table = new Table(insight.id(), insight.label(), attributes())
                    .globalRow(globalRow(cube))
                    .rows(rows(cube, node));

            return Renderer.render(table);
        } catch (TableChartException e) {
            Logger.error(e);
            return "";
        }
    }

    protected List<Attribute> attributes() {
        return stream(insight.indicators()).map(i -> new Attribute(i, i)).collect(Collectors.toList());
    }

    protected Row globalRow(Cube cube) {
        return insight.maxItems() == Integer.MAX_VALUE ? new Row("").cells(tableCells(cube.cell(""))) : null;
    }

    protected List<Row> rows(Cube cube, Node node) {
        return insight.dimensions().length > 1 ? rowsCubeCells(cube, node) : rowSlices(cube, node);
    }

    private List<Row> rowsCubeCells(Cube cube, Node node) {
        return cube.cells().stream()
                .filter(Objects::nonNull)
                .filter(this::containAllDimensions)
                .filter(this::slicesMatchLevel)
                .map(c -> row(c.toString(), cube, node, c.slices().toArray(Slice[]::new)))
                .filter(Objects::nonNull)
                .sorted(sortMethodOf(insight.order()))
                .limit(insight.maxItems())
                .collect(Collectors.toCollection(LinkedList::new));
    }

    private List<Row> rowSlices(Cube cube, Node node) {
        List<Slice> slices = (isNavigable(node) && isNotCustomLevel()) ? slices(cube, node) : slices(cube);
        return slices.stream()
                .map(slice -> row(slice.name(), cube, node, slice))
                .filter(Objects::nonNull)
                .sorted(sortMethodOf(insight.order()))
                .limit(insight.maxItems())
                .collect(Collectors.toCollection(LinkedList::new));
    }

    private Row row(String name, Cube cube, Node node, Slice... slices) {
        List<Table.Cell> cells = tableCells(cube.cell(name));
        if (areInvalid(cells)) return null;

        return new Row(insight.translate(name))
                .cells(cells)
                .onClick(action(name, node))
                .onMicrosite(micrositeAction(node, slices));
    }

    private List<Table.Cell> tableCells(Cell cell) {
        return stream(insight.indicators())
                .map(i -> CubesHelper.indicatorOf(report, cell, i))
                .map(this::tableCell)
                .collect(Collectors.toList());
    }

    private Table.Cell tableCell(Cube.Indicator indicator) {
        return new Table.Cell(indicator.name(), value(indicator), indicator.units()).format(formatter(indicator));
    }

    private boolean areInvalid(List<Table.Cell> cells) {
        return cells.stream()
                .allMatch(cell -> {
                    Object object = cell.value();
                    if (cell.isNumeric()) {
                        Double value = (Double) object;
                        return value == 0 || value.equals(Double.NaN) || value.longValue() == Long.MAX_VALUE || value.longValue() == Long.MIN_VALUE;
                    }
                    return object == null ||
                            object == LocalDate.MAX || object == LocalDate.MIN ||
                            object == LocalTime.MAX || object == LocalTime.MIN;
                }
        );
    }

    private boolean containAllDimensions(Cell cell) {
        List<String> cellDimensions = cell.slices().stream().map(s -> s.dimension().name().toLowerCase()).collect(Collectors.toList());
        return stream(insight.dimensions()).allMatch(d -> cellDimensions.contains(d.toLowerCase()));
    }

    private List<Slice> slices(Cube cube, Node node) {
        return node.children().stream()
                .map(Node::name)
                .map(cube::cell)
                .filter(Objects::nonNull)
                .flatMap(c -> c.slices().stream())
                .collect(Collectors.toList());
    }

    private List<Slice> slices(Cube cube) {
        return cube.dimensions().stream()
                .flatMap(d -> d.slices(level()).stream())
                .collect(Collectors.toList());
    }

    private boolean slicesMatchLevel(Cell cell) {
        return isNotCustomLevel() || cell.slices().stream().allMatch(this::sliceMatchLevel);
    }

    private boolean sliceMatchLevel(Slice slice) {
        return slice.dimension().levels() <= 1 || slice.level() == insight.level();
    }

    private int level() {
        return insight.level() != null ? insight.level() : MaxLevel;
    }

    private String action(String name, Node node) {
        return isNavigable(node) && !isCustomNode() ? openNodeAction(name) : null;
    }

    public static String openNodeAction(String name) {
        return "javascript:openNode('" + name + "')";
    }

    private String micrositeAction(Node node, Slice... slices) {
        return microsite.action(node, slices);
    }

    private Formatter formatter(Cube.Indicator indicator) {
        return report.formatter(insight.format(indicator.name()));
    }

    private Object value(Cube.Indicator indicator) {
        Object value = indicator.value();
        if (value instanceof LocalTime || value instanceof LocalDate) return value;
        return CubesHelper.doubleValueOf(indicator);
    }

    private boolean isNavigable(Node node) {
        return node.dimension() != null && insight.dimensions().length == 1 && insight.dimensions()[0].equalsIgnoreCase(node.dimension());
    }

    private boolean isNotCustomLevel() {
        return insight.level() == null;
    }

    private boolean isCustomNode() {
        return insight.node() != null;
    }
}
