package io.intino.sumus.reporting.builders;

import io.intino.sumus.engine.Cube;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.reporting.Dashboard;
import io.intino.sumus.reporting.Node;
import io.intino.sumus.reporting.builders.schemas.ColumnChart;
import io.intino.sumus.reporting.builders.schemas.ColumnChart.Serie;
import io.intino.sumus.reporting.builders.templates.Renderer;
import io.intino.sumus.reporting.helpers.CubesHelper;

import java.util.*;
import java.util.stream.Collectors;

public class StackedColumnChartBuilder extends ColumnChartBuilder {

	public StackedColumnChartBuilder(Dashboard.Report report, Dashboard.Insight insight, MicrositeActionBuilder microsite) {
		super(report, insight, microsite);
	}

	@Override
	public String build(Cube cube, Node node) {
		return insight.dimensions().length == 2 ?
				buildStackedColumn(cube, node) :
				buildColumnChart(cube, node);
	}

	private String buildColumnChart(Cube cube, Node node) {
		return new ColumnChartBuilder(report, insight, microsite).build(cube, node);
	}

	private String buildStackedColumn(Cube cube, Node node) {
		return Renderer.render(stackedColumn(cube, node));
    }

	private ColumnChart stackedColumn(Cube cube, Node node) {
		String indicator = insight.indicators()[0];
		Map<String, Serie> seriesMap = new HashMap<>();

		for (Slice categorySlice : categorySlices(cube, node)) {
			String category = categorySlice.toString();

			for (Slice seriesSlice : seriesSlices(cube, node)) {
				double value = valueOf(cube, categorySlice, seriesSlice, indicator);
				if (isInvalid(value)) continue;

				String serie = seriesSlice.toString();
				seriesMap.putIfAbsent(serie, new Serie(serie));
				seriesMap.get(serie).addValue(category, value);
			}
		}
		List<Serie> series = new ArrayList<>(seriesMap.values());
		return new ColumnChart(insight.id(), insight.label(), insight.order())
				.showDataLabels(insight.showDataLabels())
				.stacked(true)
				.units(units(cube))
				.plotLines(plotLines())
				.series(series);
	}

	private List<Slice> categorySlices(Cube cube, Node node) {
		return slices(cube, node, insight.dimensions()[0]);
	}

	private List<Slice> seriesSlices(Cube cube, Node node) {
		return slices(cube, node, insight.dimensions()[1]);
	}

	private double valueOf(Cube cube, Slice slice1, Slice slice2, String indicator) {
		Cube.Cell cell = cube.cell(slice1 + "-" + slice2);
        return CubesHelper.doubleValueOf(CubesHelper.indicatorOf(report, cell, indicator));
	}

	private List<Slice> slices(Cube cube, Node node, String dimension) {
		return isNavigable(node, dimension) ? slices(cube, node) : slices(cube, dimension);
	}

	private List<Slice> slices(Cube cube, Node node) {
		return node.children().stream()
				.map(Node::name)
				.map(cube::cell)
				.filter(Objects::nonNull)
				.flatMap(c -> c.slices().stream())
				.sorted(Comparator.comparing(Slice::toString))
				.collect(Collectors.toList());
	}

	private List<Slice> slices(Cube cube, String dimension) {
		return cube.dimensions().stream()
				.filter(d -> d.name().equalsIgnoreCase(dimension))
				.flatMap(d -> d.slices().stream())
				.sorted(Comparator.comparing(Slice::toString))
				.collect(Collectors.toList());
	}

	private String units(Cube cube) {
		Cube.Cell cell = cube.cell("");
		if (cell == null) return "";

		List<String> units = cell.indicators().stream().map(Cube.Indicator::units).collect(Collectors.toList());
		if (units.isEmpty() || units.contains(null)) return "";

		String unit = units.get(0);
		return units.stream().allMatch(unit::equalsIgnoreCase) ? unit : "";
	}
}
