package io.intino.sumus.engine.model;

import io.intino.sumus.engine.helpers.Json;

import java.io.Serializable;
import java.util.*;
import java.util.function.Predicate;

public abstract class DimensionDefinition implements Serializable {

	public enum Type {
		Numerical, Categorical
	}

	private final Type type;
	private final String name;
	private final AttributeDefinition attribute;
	private final Map<String, Predicate<?>> categories;
	private Map<String, String> categoryValue = new HashMap<>();

	public DimensionDefinition(Type type, String name, AttributeDefinition attribute, Map<String, Predicate<?>> categories) {
        this.type = type;
        this.name = name;
		this.attribute = attribute;
		this.categories = Collections.unmodifiableMap(categories);
	}

	public DimensionDefinition categoryRange(Map<String, String> categoryRange) {
		this.categoryValue = categoryRange;
		return this;
	}

	public AttributeDefinition attribute() {
		return attribute;
	}

	public String name() {
		return name;
	}

	public Classifier classifier() {
		return new Classifier() {
			@Override
			public List<String> categories() {
				return new ArrayList<>(categories.keySet());
			}

			@Override
			@SuppressWarnings("unchecked")
			public <T> Predicate<T> predicateOf(String category) {
				return (Predicate<T>) categories.get(category);
			}

			@Override
			public boolean classify(String category, Object value) {
				return predicateOf(category).test(value);
			}
		};
	}

	public boolean isNumeric() {
		return type == Type.Numerical;
	}

	public Map<String, Predicate<?>> categories() {
		return categories;
	}

	@Override
	public String toString() {
		return Json.toJsonPretty(this);
	}

	public static class Numerical extends DimensionDefinition {
		public Numerical(String name, AttributeDefinition attribute, Map<String, Predicate<?>> categories) {
			super(Type.Numerical, name, attribute, categories);
		}
	}

	public static class Categorical extends DimensionDefinition {
		public Categorical(String name, AttributeDefinition attribute, Map<String, Predicate<?>> categories) {
			super(Type.Categorical, name, attribute, categories);
		}
	}
}
