package io.intino.sumus.engine.model;

import io.intino.sumus.engine.helpers.Json;

import java.util.HashMap;
import java.util.Map;

import static io.intino.sumus.engine.helpers.Formatters.firstUpperCase;

public abstract class AttributeDefinition {
	private static final AttributeBuilder nullBuilder = (n, p) -> null;
	private static final Map<String, AttributeBuilder> builders = new HashMap<>() {{
		put(Category.class.getSimpleName(), (n, p) -> new Category(n));
		put(Int.class.getSimpleName(), (n, p) -> new Int(n));
		put(Int8.class.getSimpleName(), (n, p) -> new Int8(n));
		put(Int16.class.getSimpleName(), (n, p) -> new Int16(n));
		put(Int32.class.getSimpleName(), (n, p) -> new Int32(n));
		put(Int64.class.getSimpleName(), (n, p) -> new Int64(n));
		put(Real.class.getSimpleName(), (n, p) -> new Real(n));
		put(Real32.class.getSimpleName(), (n, p) -> new Real32(n));
		put(Real64.class.getSimpleName(), (n, p) -> new Real64(n));
		put(Label.class.getSimpleName(), (n, p) -> new Label(n));
		put(Date.class.getSimpleName(), (n, p) -> new Date(n));
		put(Time.class.getSimpleName(), (n, p) -> new Time(n));
		put(Key.class.getSimpleName(), (n, p) -> new Key(n, p.length == 0 ? "" : p[0]));
		put(Url.class.getSimpleName(), (n, p) -> p.length != 1 ? null : new Url(n, p[0]));
	}};

	private final String name;
	private final Type type;

	public static AttributeDefinition create(String type, String name, String... parameters) {
		return builders.getOrDefault(firstUpperCase(type), nullBuilder).build(name, parameters);
	}

	public AttributeDefinition(String name) {
		this.name = name;
		this.type = Type.valueOf(getClass().getSimpleName().toLowerCase());
	}

	public AttributeDefinition(String name, Type type) {
		this.name = name;
		this.type = type;
	}

	public Type type() {
		return type;
	}

	public boolean isNumeric() {
		return type.isNumeric();
	}

	public boolean isOrdinal() {
		return type.isOrdinal();
	}

	public String name() {
		return name;
	}

	public boolean isReal() {
		return type().isReal();
	}

	@Override
	public String toString() {
		return Json.toJsonPretty(this);
	}

	public static class Category extends AttributeDefinition {
		public Category(String name) {
			super(name);
		}
	}

	public static class Int extends Int64 {
		public Int(String name) {
			super(name);
		}
	}

	public static class Int8 extends AttributeDefinition {
		public Int8(String name) {
			super(name);
		}
	}

	public static class Int16 extends AttributeDefinition {
		public Int16(String name) {
			super(name);
		}
	}

	public static class Int32 extends AttributeDefinition {
		public Int32(String name) {
			super(name);
		}
	}

	public static class Int64 extends AttributeDefinition {
		public Int64(String name) {
			super(name, Type.int64);
		}
	}

	public static class Real extends Real64 {
		public Real(String name) {
			super(name);
		}
	}

	public static class Real32 extends AttributeDefinition {
		public Real32(String name) {
			super(name);
		}
	}

	public static class Real64 extends AttributeDefinition {
		public Real64(String name) {
			super(name, Type.real64);
		}
	}

	public static class Date extends AttributeDefinition {
		public Date(String name) {
			super(name);
		}
	}

	public static class Time extends AttributeDefinition {
		public Time(String name) {
			super(name);
		}
	}

	public static class Label extends AttributeDefinition {
		public Label(String name) {
			super(name);
		}
	}

	public static class Key extends AttributeDefinition {
		private final String ledgerJoin;

		public Key(String name, String ledgerJoin) {
			super(name);
			this.ledgerJoin = ledgerJoin;
		}

		public String ledgerJoin() {
			return ledgerJoin;
		}
	}

	public static class Url extends AttributeDefinition {
		private final String path;

		public Url(String name, String path) {
			super(name);
			this.path = path;
		}

		public String path() {
			return path;
		}
	}

	private interface AttributeBuilder {
		AttributeDefinition build(String name, String... parameters);
	}

	public enum Type {

		int8, int16, int32, int64,
		real32, real64,
		label, category, date, time, key, url;

		public boolean isNumeric() {
			return isInteger() || isReal();
		}

		public boolean isReal() {
			return this == real32 || this == real64;
		}

		public boolean isInteger() {
			return this == int8 || this == int16 || this == int32 || this == int64;
		}

		public boolean isDate() {
			return this == date;
		}

		public boolean isTime() {
			return this == time;
		}

		public boolean isOrdinal() {
			return isNumeric() || this == date;
		}
	}
}
