package io.intino.sumus.engine.ledgers.columnar.columns;

import io.intino.sumus.engine.Attribute;
import io.intino.sumus.engine.dimensions.Category;
import io.intino.sumus.engine.ledgers.columnar.Column;
import io.intino.sumus.engine.model.AttributeDefinition;

import java.time.LocalDate;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import static java.util.Collections.emptyList;

public class DataColumn implements Column {

    public final Attribute attribute;
    public final Object[] values;
    private Object min;
    private Object max;
    private boolean hasNA;

    public DataColumn(Attribute attribute, String... values) {
        this.attribute = attribute;
        this.values = map(values, function(values));
    }

    private Comparator<Object> comparatorOf(AttributeDefinition.Type type) {
        switch (type) {
            case int8:
            case int16:
            case int32:
            case int64:
            case date:
            case time:
                return Comparator.comparingLong(a -> (long) a);
            case real32:
            case real64:
                return Comparator.comparingDouble(a -> (double) a);
        }
        return (a, b) -> 0;
    }

    private Function<String, Object> function(String[] values) {
        switch (attribute.type()) {
            case category:
                return null;
            case int8:
            case int16:
            case int32:
            case int64:
                return Long::parseLong;
            case real32:
            case real64:
                return Double::parseDouble;
            case date:
                return DataColumn.this::toEpoch;
            case time:
                return DataColumn.this::toSecondDay;
        }
        return s -> s;
    }

    private Object[] map(String[] values, Function<String, Object> function) {
        if(attribute.type() == AttributeDefinition.Type.category) return mapCategories(values);
        final int size = values.length;
        final Object[] result = new Object[size];
        for(int i = 0;i < size;i++) {
            String raw = clean(values[i]);
            Object value = raw.isEmpty() ? null : catching(() -> function.apply(raw));
            hasNA |= value == null;
            result[i] = value;
        }
        return result;
    }

    private Category[] mapCategories(String[] values) {
        int size = values.length;
        Map<String, Category> categoryMap = new HashMap<>();
        Category[] result = new Category[size];
        for(int i = 0;i < size;i++) {
            String str = clean(values[i]);
            if(!categoryMap.containsKey(str)) update(categoryMap, split(str));
            Category category = categoryMap.get(str);
            hasNA |= category == null;
            result[i] = category;
        }
        return result;
    }

    private static String clean(String v) {
        return v != null ? v.trim() : "";
    }

    private static List<String> split(String value) {
        if (value.isEmpty()) return emptyList();
        List<String> result = new ArrayList<>();
        int i = value.indexOf('.');
        while (i > 0) {
            result.add(value.substring(0, i));
            i = value.indexOf('.', i + 1);
        }
        result.add(value);
        return result;
    }

    private static void update(Map<String, Category> categories, List<String> labels) {
        Category parent = null;
        for (String label : labels) {
            if (!categories.containsKey(label))
                categories.put(label, new Category(categories.size() + 1, label, parent));
            parent = categories.get(label);

        }
    }

    private void setRange() {
        Comparator<Object> comparator = comparatorOf(attribute.type());
        for (Object value : values) {
            if (value == null) continue;
            if (min == null || comparator.compare(value, min) < 0) min = value;
            if (max == null || comparator.compare(value, max) > 0) max = value;
        }
    }

    @Override
    public String name() {
        return attribute.name();
    }

    @Override
    public AttributeDefinition.Type type() {
        return attribute.type();
    }

    public int size() {
        return values.length;
    }

    @Override
    public Attribute attribute() {
        return attribute;
    }

    @Override
    public Object value(int idx) {
        return values[idx];
    }

    @Override
    public boolean hasNA() {
        return hasNA;
    }

    @Override
    public Stream<?> uniques() {
        return Arrays.stream(values)
                .distinct()
                .filter(Objects::nonNull);
    }

    @Override
    public Object min() {
        if (min == null) setRange();
        return min;
    }

    @Override
    public Object max() {
        if (max == null) setRange();
        return max;
    }

    private Object catching(Callable<Object> function) {
        try {
            return function.call();
        } catch (Exception e) {
            return null;
        }
    }

    private static final DateTimeFormatter DateFormatter = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final DateTimeFormatter TimeFormatter = DateTimeFormatter.ofPattern("HHmmss");

    private Object toEpoch(String value) {
        if (value == null) return null;
        return LocalDate.parse(trim(value), DateFormatter).toEpochDay();
    }

    private Object toSecondDay(String value) {
        if (value == null) return null;
        return LocalTime.parse(trim(value), TimeFormatter).toSecondOfDay();
    }

    private static final Pattern TrimPattern = Pattern.compile("[-/: .]");
    private static String trim(String value) {
        return TrimPattern.matcher(value).replaceAll("");
    }

    @Override
    public String toString() {
        return type().name() + " " + name();
    }
}
