package io.intino.sumus.engine.ledgers.columnar;

import io.intino.sumus.engine.*;
import io.intino.sumus.engine.builders.CubeBuilder;
import io.intino.sumus.engine.dimensions.*;
import io.intino.sumus.engine.io.TsvLedgerReader;
import io.intino.sumus.engine.ledgers.columnar.columns.DataColumn;
import io.intino.sumus.engine.model.AttributeDefinition;
import io.intino.sumus.engine.model.DimensionDefinition;
import io.intino.sumus.engine.model.LedgerDefinition;

import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.function.Predicate;

import static io.intino.sumus.engine.model.AttributeDefinition.Type.category;
import static io.intino.sumus.engine.model.AttributeDefinition.Type.date;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

public class ColumnarLedger implements Ledger {

	public final LedgerDefinition definition;
	public final List<Column> columns;
	public final List<Dimension> dimensions;
	private final Map<String, Column> columnMap = new HashMap<>();
	private int size;

	public ColumnarLedger(LedgerDefinition definition) {
		this.definition = definition;
		this.columns = new ArrayList<>(definition.attributes.size());
		this.dimensions = new ArrayList<>(definition.attributes.size());
	}

	@Override
	public LedgerDefinition definition() {
		return definition;
	}

	public ColumnarLedger load(File file, String separator) throws IOException {
		return load(read(file, separator));
	}

	public ColumnarLedger load(String[][] data) {
		loadingWith(transpose(data));
		return this;
	}

	private void loadingWith(String[][] data) {
		ExecutorService threadPool = SumusEngine.createThreadPool();
		for (int i = 0; i < definition.attributes.size(); i++) {
			Attribute attribute = attributeOf(i);
			String[] values = data[i];
			threadPool.execute(() -> add(new DataColumn(attribute, values)));
		}
		waitFor(threadPool);
		sortColumnList();
	}

	private void waitFor(ExecutorService threadPool) {
		try {
			threadPool.shutdown();
			threadPool.awaitTermination(SumusEngine.TIMEOUT_AMOUNT.get(), SumusEngine.TIMEOUT_UNIT.get());
		} catch (InterruptedException e) {
			throw new RuntimeException(e);
		}
	}

	private void sortColumnList() {
		columns.clear();
		for(AttributeDefinition attrib : definition.attributes) {
			columns.add(column(attrib.name()));
		}
	}

	private String[][] read(File file, String separator) throws IOException {
		return TsvLedgerReader.read(file, separator);
	}

	private Attribute attributeOf(int i) {
		return new ColumnarAttribute(this.definition.attributes.get(i));
	}

	private String[][] transpose(String[][] rows) {
		String[][] columns = initColumns(rows.length);
		for (int i = 0; i < rows.length; i++) {
			String[] row = rows[i];
			for (int j = 0; j < row.length; j++) columns[j][i] = row[j];
		}
		return columns;
	}

	private String[][] initColumns(int size) {
		return new String[definition.attributes.size()][size];
	}

	public synchronized ColumnarLedger add(Column column) {
		if (column == null) return this;
		if (columnMap.isEmpty()) size = column.size();
		this.columns.add(column);
		this.dimensions.addAll(dimensionsOf(column));
		this.columnMap.put(column.name(), column);
		return this;
	}

	public void removeColumnIf(Predicate<Column> condition) {
		List<Column> columnsToRemove = columns.stream().filter(condition).collect(toList());
		for (Column column : columnsToRemove) {
			columns.remove(column);
			columnMap.remove(column.name());
		}
	}

	private List<Dimension> dimensionsOf(Column column) {
		if (column.type() == category) return categoricalDimensionsOf(column);
		if (column.type() == date) return dateDimensionsOf(column);
		if (column.type().isNumeric()) return numericDimensionsOf(column);
		return emptyList();
	}

	private List<Dimension> categoricalDimensionsOf(Column column) {
		return List.of(new CategoricalDimension(column));
	}

	private List<Dimension> dateDimensionsOf(Column column) {
		return List.of(
				new YearDimension(column),
				new MonthOfYearDimension(column),
				new DayOfWeekDimension(column),
				new YearMonthDimension(column)
		);
	}

	private List<Dimension> numericDimensionsOf(Column column) {
		return Arrays.stream(column.attribute().dimensions())
				.map(d -> dimension(column, d))
				.collect(toList());
	}

	private Dimension dimension(Column column, DimensionDefinition definition) {
		return new NumericalDimension(column, definition.name(), definition.classifier());
	}

	@Override
	public Query cube() {
		return new Query() {
			private List<Dimension> dimensions = emptyList();
			private Filter filter = Filter.None;

			@Override
			public Query filter(Filter filter) {
				this.filter = filter;
				return this;
			}

			@Override
			public Query dimensions(List<Dimension> dimensions) {
				this.dimensions = dimensions;
				return this;
			}

			@Override
			public Cube build() {
				return new CubeBuilder(ColumnarLedger.this, filter, dimensions).build();
			}
		};
	}

	public Fact fact(int idx) {
		return new ColumnarFact(idx);
	}

	@Override
	public Iterable<Fact> facts(Filter filter) {
		return () -> new FactIterator(filter);
	}

	@Override
	public List<Column> columns(String name) {
		return List.of(column(name));
	}

	@Override
	public int size() {
		return size;
	}

	@Override
	public List<Attribute> attributes() {
		return columns.stream().map(Column::attribute).collect(toList());
	}

	@Override
	public List<Dimension> dimensions() {
		return dimensions;
	}

	public Column column(String name) {
		return columnMap.getOrDefault(name, Column.Null);
	}

	private class FactIterator implements Iterator<Fact> {
		private final Filter filter;
		private int idx = 0;
		private Fact fact;

		public FactIterator(Filter filter) {
			this.filter = filter;
			this.fact = nextFact();
		}

		@Override
		public boolean hasNext() {
			return fact != null;
		}

		@Override
		public Fact next() {
			Fact result = fact;
			fact = nextFact();
			return result;
		}

		private Fact nextFact() {
			while (idx < size)
				if (filter.accepts(idx++)) return fact(idx - 1);
			return null;
		}

		private Fact fact(int idx) {
			return new ColumnarFact(idx);
		}
	}

	private class ColumnarFact implements Fact {

		private final int idx;
		private final Map<String, Object> values = new HashMap<>(attributes().size());

		public ColumnarFact(int idx) {
			this.idx = idx;
			for(Attribute attribute : attributes()) {
				values.put(attribute.name(), column(attribute.name()).value(idx));
			}
		}

		@Override
		public int idx() {
			return idx;
		}

		@Override
		public List<Attribute> attributes() {
			return ColumnarLedger.this.attributes();
		}

		@Override
		public Object value(String attribute) {
			return values.get(attribute);
		}

		@Override
		public String toString() {
			return attributes().stream()
					.map(a -> a.name() + ":" + format(value(a.name())))
					.collect(joining(","));
		}

		private Object format(Object value) {
			return value == null ? "" : value.toString();
		}

	}

	public class ColumnarAttribute extends AbstractAttribute {

		public ColumnarAttribute(AttributeDefinition definition) {
			super(definition);
		}

		@Override
		public AttributeDefinition definition() {
			return definition;
		}

		@Override
		protected LedgerDefinition ledgerDefinition() {
			return ColumnarLedger.this.definition;
		}
	}
}
