package io.intino.sumus.engine.dimensions;

import io.intino.sumus.engine.Lookup;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.engine.SumusException;

import java.time.LocalDate;
import java.time.Month;
import java.util.Arrays;
import java.util.List;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static io.intino.sumus.engine.model.AttributeDefinition.Type.date;
import static java.util.stream.Collectors.toList;

public class YearMonthDimension extends AbstractDimension {

	public YearMonthDimension(Lookup lookup) {
		super(lookup);
		this.slices.addAll(buildSlices());
		if (lookup.hasNA()) this.slices.add(new DimensionSlice());
	}

	public String name() {
		return lookup.name() + "-year-month";
	}

	private List<Slice> buildSlices() {
		return years().boxed().flatMap(this::slicesOf).collect(toList());
	}

	private Stream<Slice> slicesOf(int year) {
		return Arrays.stream(Month.values()).map(month -> sliceOf(year, month));
	}

	private DimensionSlice sliceOf(int year, Month month) {
		return new DimensionSlice(year + "-" + String.format("%02d", month.getValue()), v -> match(year, month, v));
	}

	@Override
	protected void check() {
		if (lookup.type() == date) return;
		throw new SumusException("YearMonth dimension must use a date column");
	}

	private boolean match(int year, Month month, Object value) {
		return value instanceof Long && LocalDate.ofEpochDay((long) value).getMonth() == month && LocalDate.ofEpochDay((long) value).getYear() == year;
	}

	private IntStream years() {
		try {
			int min = LocalDate.ofEpochDay((Long) lookup.min()).getYear();
			int max = LocalDate.ofEpochDay((Long) lookup.max()).getYear();
			return IntStream.range(min, max + 1);
		} catch (Exception e) {
			return IntStream.empty();
		}
	}
}
