package io.intino.sumus.engine.dimensions;

import io.intino.sumus.engine.Lookup;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.engine.SumusException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static io.intino.sumus.engine.model.AttributeDefinition.Type.category;
import static java.util.stream.Collectors.toList;

public class CategoricalDimension extends AbstractDimension {

	public CategoricalDimension(Lookup lookup) {
		super(lookup);
		this.slices.addAll(buildSlices());
		if (lookup.hasNA()) this.slices.add(new DimensionSlice());
	}

	@Override
	protected void check() {
		if (lookup.type() == category) return;
		throw new SumusException("Categorical dimension must use a categorical column");
	}

	private List<Slice> buildSlices() {
		return buildSlices(categories());
	}

	private List<Category> categories() {
		return lookup.uniques()
				.flatMap(v -> all(category(v)).stream())
				.distinct()
				.collect(toList());
	}

	private List<Slice> buildSlices(List<Category> categories) {
		Map<Category, DimensionSlice> slices = new HashMap<>();
		for (Category category : categories)
			slices.put(category, sliceOf(category, slices.get(category.parent)));
		return categories.stream().map(slices::get).sorted(this::compare).collect(toList());
	}

	private DimensionSlice sliceOf(Category category, DimensionSlice parent) {
		return new DimensionSlice(category.label, v -> v instanceof Category && match((Category) v, category), parent);
	}

	private boolean match(Category value, Category category) {
		return value != null && (value == category || match(value.parent, category));
	}

	private int compare(Slice a, Slice b) {
		return a.level() - b.level();
	}

	private List<Category> all(Category category) {
		if (category == null) return new ArrayList<>();
		List<Category> result = all(category.parent);
		result.add(category);
		return result;
	}

	private Category category(Object v) {
		return (Category) v;
	}

	@Override
	public int levels() {
		return slices.stream().mapToInt(Slice::level).max().orElse(1);
	}

	@Override
	public List<Slice> slices(int level) {
		return slices.stream()
				.filter(s->s.level() == level || s.level() < level && !hasChildren(s))
				.collect(toList());
	}

	private boolean hasChildren(Slice slice) {
		for (Slice s : slices) {
			DimensionSlice dimensionSlice = (DimensionSlice) s;
			if (dimensionSlice.parent == slice) return true;
		}
		return false;
	}

}
