package io.intino.sumus.engine.builders.evaluators;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class ExpressionEvaluator {

    static final String OpenParenthesis = "(";
    static final String CloseParenthesis = ")";
    static final String Sum = "+";
    static final String Sub = "-";
    static final String Multiply = "*";
    static final String Divide = "/";
    static final String Modulus = "%";

    static final String NumberRegex = "-?\\d+(\\.\\d+)?";
    static final Pattern TokenPattern = Pattern.compile("\\d+(\\.\\d+)?|[-+*/%()]");
    static final Map<String, Integer> OperatorPrecedence = Map.of(
            Sum, 1,
            Sub, 1,
            Multiply, 2,
            Divide, 2,
            Modulus, 2
    );

    public static double evaluate(String formula) {
        return evaluate(toPostfix(tokensOf(formula)));
    }

    private static double evaluate(List<String> postfixTokens) {
        Deque<Double> stack = new ArrayDeque<>();
        for (String token : postfixTokens) {
            if (isNumber(token)) stack.push(Double.parseDouble(token));
            else {
                double b = stack.pop();
                double a = stack.pop();
                switch (token) {
                    case Sum:
                        stack.push(a + b);
                        break;
                    case Sub:
                        stack.push(a - b);
                        break;
                    case Multiply:
                        stack.push(a * b);
                        break;
                    case Divide:
                        stack.push(b != 0 ? (a / b) : 0);
                        break;
                    case Modulus:
                        stack.push(a % b);
                        break;
                    default:
                        throw new IllegalArgumentException("Unknown operator: " + token);
                }
            }
        }
        return stack.pop();
    }

    private static List<String> toPostfix(List<String> tokens) {
        List<String> result = new ArrayList<>();
        Deque<String> operators = new ArrayDeque<>();
        for (int i = 0; i < tokens.size(); i++) {
            String token = tokens.get(i);
            if (isNumber(token)) result.add(token);
            else if (isOperator(token)) {
                if (isNegativeSymbol(token) && (i == 0 || !isNumber(tokens.get(i - 1)))) {
                    result.add(Sub + tokens.get(++i));
                } else {
                    while (!operators.isEmpty() && isOperator(operators.peek()) && hasMorePreference(operators.peek(), token)) {
                        result.add(operators.pop());
                    }
                    operators.push(token);
                }
            }
            else if (isOpenParenthesis(token)) operators.push(token);
            else if (isCloseParenthesis(token)) {
                while (!operators.isEmpty() && !isOpenParenthesis(operators.peek())) {
                    result.add(operators.pop());
                }
                operators.pop();
            }
        }
        while (!operators.isEmpty()) {
            result.add(operators.pop());
        }
        return result;
    }

    private static List<String> tokensOf(String formula) {
        Matcher matcher = TokenPattern.matcher(formula);
        List<String> tokens = new ArrayList<>();
        while (matcher.find()) {
            tokens.add(matcher.group());
        }
        return tokens;
    }

    private static boolean isOperatorOrOpenParenthesis(String token) {
        return isOperator(token) || isOpenParenthesis(token);
    }

    private static boolean hasMorePreference(String peekOperator, String token) {
        return OperatorPrecedence.get(peekOperator) >= OperatorPrecedence.get(token);
    }

    private static boolean isOperator(String token) {
        return OperatorPrecedence.containsKey(token);
    }

    private static boolean isOpenParenthesis(String token) {
        return OpenParenthesis.equals(token);
    }

    private static boolean isCloseParenthesis(String token) {
        return CloseParenthesis.equals(token);
    }

    private static boolean isNegativeSymbol(String token) {
        return Sub.equals(token);
    }

    private static boolean isNumber(String token) {
        return token.matches(NumberRegex);
    }
}
