package io.intino.sumus.engine.builders.deserializers;

import com.google.gson.*;
import com.google.gson.reflect.TypeToken;
import io.intino.sumus.engine.model.AttributeDefinition;
import io.intino.sumus.engine.model.DimensionDefinition;
import io.intino.sumus.engine.model.IndicatorDefinition;
import io.intino.sumus.engine.model.LedgerDefinition;
import io.intino.sumus.engine.model.LedgerDefinition.Aggregation;
import io.intino.sumus.engine.model.LedgerDefinition.Content;
import io.intino.sumus.engine.model.LedgerDefinition.Format;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class LedgerDefinitionDeserializer implements JsonDeserializer<LedgerDefinition> {

    static final Type AttributeListType = new TypeToken<List<AttributeDefinition>>() {}.getType();

    @Override
    public LedgerDefinition deserialize(JsonElement jsonElement, Type type, JsonDeserializationContext c) throws JsonParseException {
        JsonObject object = jsonElement.getAsJsonObject();

        LedgerDefinition definition = baseLedgerDefinition(c, object);

        List<AttributeDefinition> attributes = c.deserialize(object.get("attributes"), AttributeListType);
        Map<String, AttributeDefinition> attributesMap = toMap(attributes);

        attributes.forEach(definition::add);
        indicatorsOf(object, c, attributesMap).forEach(definition::add);
        dimensionsOf(object, c, attributesMap).forEach(definition::add);
        return definition;
    }

    private static LedgerDefinition baseLedgerDefinition(JsonDeserializationContext c, JsonObject object) {
        Content content = c.deserialize(object.get("content"), Content.class);
        Format format = c.deserialize(object.get("format"), Format.class);
        Aggregation aggregation = c.deserialize(object.get("aggregation"), Aggregation.class);
        String label = c.deserialize(object.get("label"), String.class);
        String description = c.deserialize(object.get("description"), String.class);
        return new LedgerDefinition(content, format, aggregation, label, description);
    }

    private static List<IndicatorDefinition> indicatorsOf(JsonObject object, JsonDeserializationContext c, Map<String, AttributeDefinition> attributesMap) {
        List<IndicatorDefinition> indicators = new ArrayList<>();
        IndicatorDefinitionDeserializer deserializer = new IndicatorDefinitionDeserializer(attributesMap);
        for (JsonElement jsonIndicator : object.getAsJsonArray("indicators")) {
            IndicatorDefinition indicator = deserializer.deserialize(jsonIndicator, IndicatorDefinition.class, c);
            indicators.add(indicator);
        }
        return indicators;
    }

    private static List<DimensionDefinition> dimensionsOf(JsonObject object, JsonDeserializationContext c, Map<String, AttributeDefinition> attributesMap) {
        List<DimensionDefinition> dimensions = new ArrayList<>();
        DimensionDefinitionDeserializer deserializer = new DimensionDefinitionDeserializer(attributesMap);
        for (JsonElement jsonIndicator : object.getAsJsonArray("dimensions")) {
            DimensionDefinition dimension = deserializer.deserialize(jsonIndicator, DimensionDefinition.class, c);
            dimensions.add(dimension);
        }
        return dimensions;
    }

    private static Map<String, AttributeDefinition> toMap(List<AttributeDefinition> attributes) {
        return attributes.stream().collect(Collectors.toMap(AttributeDefinition::name, a -> a, (a, b) -> a));
    }
}
