package io.intino.sumus.engine.builders.deserializers;

import com.google.gson.*;
import io.intino.sumus.engine.model.AttributeDefinition;
import io.intino.sumus.engine.model.CompositeIndicatorDefinition;
import io.intino.sumus.engine.model.IndicatorDefinition;
import io.intino.sumus.engine.model.IndicatorDefinition.Scale;
import io.intino.sumus.engine.model.SimpleIndicatorDefinition;
import io.intino.sumus.engine.model.SimpleIndicatorDefinition.Function;

import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;

public class IndicatorDefinitionDeserializer implements JsonDeserializer<IndicatorDefinition> {

    private final Map<String, AttributeDefinition> attributes;

    public IndicatorDefinitionDeserializer(Map<String, AttributeDefinition> attributes) {
        this.attributes = attributes;
    }

    @Override
    public IndicatorDefinition deserialize(JsonElement jsonElement, Type deserializerType, JsonDeserializationContext c) throws JsonParseException {
        JsonObject object = jsonElement.getAsJsonObject();
        String type = c.deserialize(object.get("type"), String.class);
        IndicatorDefinition.Type indicatorType = IndicatorDefinition.Type.valueOf(type);

        switch (indicatorType) {
            case Simple -> {
                return simpleIndicator(object, c);
            }
            case Composite -> {
                return compositeIndicator(object, c);
            }
            default -> throw new JsonParseException("Unknown indicator definition type");
        }
    }

    private IndicatorDefinition simpleIndicator(JsonObject object, JsonDeserializationContext c) {
        AttributeDefinition attribute = attribute(object, c);
        String name = c.deserialize(object.get("name"), String.class);
        String unit = c.deserialize(object.get("unit"), String.class);
        Scale scale = c.deserialize(object.get("scale"), Scale.class);
        Integer decimals = c.deserialize(object.get("decimals"), Integer.class);
        Function function = c.deserialize(object.get("function"), Function.class);
        List<String> filteredValues = c.deserialize(object.get("filteredValues"), List.class);
        return new SimpleIndicatorDefinition(name, function, attribute, unit, scale, filteredValues, decimals);
    }

    private IndicatorDefinition compositeIndicator(JsonObject object, JsonDeserializationContext c) {
        String name = c.deserialize(object.get("name"), String.class);
        String unit = c.deserialize(object.get("unit"), String.class);
        Scale scale = c.deserialize(object.get("scale"), Scale.class);
        Integer decimals = c.deserialize(object.get("decimals"), Integer.class);
        String formula = c.deserialize(object.get("formula"), String.class);
        return new CompositeIndicatorDefinition(name, formula, unit, scale, decimals);
    }

    //Attribute definition must be the same instance defined in LedgerDefinition
    private AttributeDefinition attribute(JsonObject object, JsonDeserializationContext c) {
        AttributeDefinition attribute = c.deserialize(object.get("attribute"), AttributeDefinition.class);
        return attribute != null ? attributes.get(attribute.name()) : null;
    }
}
