package io.intino.sumus.engine.builders.deserializers;

import com.google.gson.*;
import io.intino.sumus.engine.model.AttributeDefinition;
import io.intino.sumus.engine.model.DimensionDefinition;
import io.intino.sumus.engine.model.DimensionDefinition.Numerical;

import java.lang.reflect.Type;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static io.intino.sumus.engine.model.DimensionDefinition.Categorical;
import static java.util.Objects.isNull;
import static java.util.regex.Pattern.compile;

public class DimensionDefinitionDeserializer implements JsonDeserializer<DimensionDefinition> {

    static final String Diacrit = "~";
    static final String OpenBracket = "[";
    static final String CloseBracket = "]";

    private final Map<String, AttributeDefinition> attributes;

    public DimensionDefinitionDeserializer(Map<String, AttributeDefinition> attributes) {
        this.attributes = attributes;
    }

    @Override
    public DimensionDefinition deserialize(JsonElement jsonElement, Type deserializerType, JsonDeserializationContext c) throws JsonParseException {
        JsonObject object = jsonElement.getAsJsonObject();
        String type = c.deserialize(object.get("type"), String.class);
        DimensionDefinition.Type dimensionType = DimensionDefinition.Type.valueOf(type);
        return definition(dimensionType, object, c);
    }

    private DimensionDefinition definition(DimensionDefinition.Type type, JsonObject object, JsonDeserializationContext c) {
        AttributeDefinition attribute = attribute(object, c);
        String name = c.deserialize(object.get("name"), String.class);
        Map<String, String> categoryValue = c.deserialize(object.get("categoryValue"), Map.class);

        return type == DimensionDefinition.Type.Numerical ?
                new Numerical(name, attribute, numericalCategories(attribute, categoryValue)) :
                new Categorical(name, attribute, categoricalCategories(categoryValue));
    }

    private Map<String, Predicate<?>> categoricalCategories(Map<String, String> categoryValue) {
        Map<String, Predicate<?>> categories = new LinkedHashMap<>();
        categoryValue.forEach((label, regex) -> categories.put(label, compile(regex).asMatchPredicate()));
        return categories;
    }

    private Map<String, Predicate<?>> numericalCategories(AttributeDefinition attribute, Map<String, String> categoryValue) {
        Map<String, Predicate<?>> categories = new LinkedHashMap<>();
        categoryValue.forEach((label, range) -> {
            Matcher matcher = rangePattern(range);
            if (!matcher.matches())
                throw new JsonParseException("Numerical dimension slice not well formatted. Pattern didn't match. " + range);
            categories.put(label, numericalSlicePredicate(matcher, range, attribute));
        });
        return categories;
    }

    private Predicate<?> numericalSlicePredicate(Matcher matcher, String range, AttributeDefinition attribute) throws JsonParseException {
        if (range.startsWith(OpenBracket) || matcher.group(2).equals(Diacrit))
            return leftClosedPredicate(attribute, parse(matcher.group(2)), parse(matcher.group(5)));
        if (range.endsWith(CloseBracket) || matcher.group(5).equals(Diacrit))
            return rightClosedPredicate(attribute, parse(matcher.group(0)), parse(matcher.group(1)));
        else
            throw new JsonParseException("Numerical dimension slice not well formatted. Missing closed interval. " + range);
    }

    private Predicate<?> leftClosedPredicate(AttributeDefinition attr, Long left, Long right) {
        return attr.isReal() ?
                doublePredicate(v -> isNull(left) || toDouble(v) >= left, v -> isNull(right) || toDouble(v) < right) :
                longPredicate(v -> isNull(left) || toLong(v) >= left, v -> isNull(right) || toLong(v) < right);
    }

    private Predicate<?> rightClosedPredicate(AttributeDefinition attr, Long left, Long right) {
        return attr.isReal() ?
                doublePredicate(v -> isNull(left) || toDouble(v) > left, v -> isNull(right) || toDouble(v) <= right) :
                longPredicate(v -> isNull(left) || toLong(v) > left, v -> isNull(right) || toLong(v) <= right);
    }

    private Predicate<Object> longPredicate(Predicate<Object> left, Predicate<Object> right) {
        return v -> !isNull(v) && left.test(toLong(v)) && right.test(toLong(v));
    }

    private Predicate<Object> doublePredicate(Predicate<Object> left, Predicate<Object> right) {
        return v -> !isNull(v) && left.test(toDouble(v)) && right.test(toDouble(v));
    }

    private static long toLong(Object v) {
        return ((Number) v).longValue();
    }

    private static double toDouble(Object v) {
        return ((Number) v).doubleValue();
    }

    private static Long parse(String s) {
        try {
            if (s == null) return null;
            return Long.parseLong(s.trim());
        } catch (NumberFormatException e) {
            return null;
        }
    }

    private static Matcher rangePattern(String range) {
        String patternStr = "([\\[(])((\\d+)|~),\\s?((\\d+)|~)(]|\\))";
        Pattern pattern = compile(patternStr);
        return pattern.matcher(range);
    }

    //Attribute definition must be the same instance defined in LedgerDefinition
    private AttributeDefinition attribute(JsonObject object, JsonDeserializationContext c) {
        AttributeDefinition attribute = c.deserialize(object.get("attribute"), AttributeDefinition.class);
        return attribute != null ? attributes.get(attribute.name()) : null;
    }
}
