package io.intino.sumus.engine.builders.accumulators;

import io.intino.sumus.engine.Cube;

import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;

import static io.intino.sumus.engine.Cube.indicator;

public class DateAccumulator extends BuilderAccumulator {

	private final int[] count = new int[threadCount];
	private final long[] min = new long[threadCount];
	private final long[] max = new long[threadCount];

	public DateAccumulator(String name) {
		super(name);
		Arrays.fill(this.min, Long.MAX_VALUE);
		Arrays.fill(this.max, Long.MIN_VALUE);
	}

	@Override
	public void add(Object value) {
		if (value == null) return;
		final int t = threadIndex();
		this.count[t]++;
		this.min[t] = Math.min(this.min[t], (long) value);
		this.max[t] = Math.max(this.max[t], (long) value);
	}

	@Override
	public List<Cube.Indicator> indicators() {
		long count = Arrays.stream(this.count).sum();
		long min = Arrays.stream(this.min).min().orElse(Long.MAX_VALUE);
		long max = Arrays.stream(this.max).max().orElse(Long.MIN_VALUE);

		return List.of(
				indicator("count(" + name + ")", count),
				indicator("min(" + name + ")", minDate(min)),
				indicator("max(" + name + ")", maxDate(max))
		);
	}

	private LocalDate minDate(long min) {
		return min == Long.MAX_VALUE ? LocalDate.MAX : LocalDate.ofEpochDay(min);
	}

	private LocalDate maxDate(long max) {
		return max == Long.MIN_VALUE ? LocalDate.MIN : LocalDate.ofEpochDay(max);
	}
}
