package io.intino.sumus.engine.builders.accumulators;

import io.intino.sumus.engine.Cube;
import io.intino.sumus.engine.dimensions.Category;

import java.util.*;
import java.util.stream.Collectors;

import static io.intino.sumus.engine.Cube.indicator;

public class CountAccumulator extends BuilderAccumulator {

	private final CountMap[] counts = new CountMap[threadCount];

	public CountAccumulator(String name) {
		super(name);
		for(int i = 0;i < counts.length;i++) counts[i] = new CountMap();
	}

	@Override
	public void add(Object value) {
		final int t = threadIndex();
		increment(t, "");
		if (value instanceof Category) increment(t, ((Category)value).label);
	}

	private void increment(int t, String key) {
		CountMap counts = this.counts[t];
		counts.put(key, counts.getOrDefault(key, 0) + 1);
	}

	@Override
	public List<Cube.Indicator> indicators() {
		Map<String, Integer> counts = countsMap();
		long total = total();

		return counts.entrySet().stream()
				.map(e -> indicatorsOf(segmentOf(e.getKey()),e.getValue(), total))
				.flatMap(Collection::stream)
				.collect(Collectors.toList());
	}

	public Map<String, Integer> countsMap() {
		Map<String, Integer> counts = new HashMap<>();
		for(CountMap c : this.counts) {
			c.forEach((k, v) -> counts.compute(k, (kk, vv) -> vv == null ? v : vv + v));
		}
		return counts;
	}

	private String segmentOf(String key) {
		return key.isEmpty() ? "" : " with " + key;
	}

	private List<Cube.Indicator> indicatorsOf(String segment, int count, long total) {
		return List.of(
				indicator("count(" + this.name + ")" + segment, count),
				indicator("ratio(" + this.name + ")" + segment, (double) count / total)
		);
	}

	public int total() {
		return Arrays.stream(this.total).sum();
	}

	private static class CountMap extends HashMap<String, Integer> {
		public CountMap() {
			super(1024);
		}
	}
}
