package io.intino.sumus.engine.ledgers.composite;

import io.intino.sumus.engine.Dimension;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.engine.helpers.IgnoreCaseMap;
import io.intino.sumus.model.AttributeDefinition;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;

public class CompositeDimension implements Dimension {
	private final List<Dimension> dimensions;
	private final List<Slice> slices;
	private final int[] offsets;

	public CompositeDimension(List<Dimension> dimensions, int[] offsets) {
		this.dimensions = dimensions;
		this.slices = new ArrayList<>();
		this.offsets = offsets;
	}
	@Override
	public String name() {
		return dimensions.get(0).name();
	}

	@Override
	public AttributeDefinition.Type type() {
		return dimensions.get(0).type();
	}

	@Override
	public boolean hasNA() {
		for (Dimension dimension : dimensions)
			if (dimension.hasNA()) return true;
		return false;
	}

	@Override
	public List<? extends Slice> slices() {
		if (slices.isEmpty()) slices.addAll(buildSlices());
		return slices;
	}

	@Override
	public boolean isOrdinal() {
		return false; // TODO
	}

	public List<Slice> slices(int level) {
		return new ArrayList<>(dimensions.stream()
				.flatMap(d -> d.slices(level).stream())
				.collect(Collectors.toMap(Slice::name, s -> s, (a, b) -> b, IgnoreCaseMap::new))
				.values());
	}

	private List<Slice> buildSlices() {
		List<String> names = sliceNames();
		Map<String,Slice> slices = new HashMap<>();
		for (String name : names)
			slices.put(name, new CompositeSlice(find(name), offsets, slices.get(parentOf(name))));
		return names.stream().map(slices::get).collect(toList());
	}

	private String parentOf(String name) {
		int i = name.lastIndexOf('.');
		return i > 0 && name.charAt(i - 1) != '.' ? name.substring(0, i) : null;
	}

	private List<Slice> find(String name) {
		return dimensions.stream().map(d->find(name,d)).collect(toList());
	}

	private Slice find(String name, Dimension dimension) {
		return dimension.slices().stream()
				.filter(s->s.name().equals(name))
				.findFirst()
				.orElse(null);
	}

	private List<String> sliceNames() {
		return dimensions.stream()
				.flatMap(d->d.slices().stream())
				.map(Slice::name)
				.distinct()
				.collect(toList());
	}

	@Override
	public String toString() {
		return name();
	}
}
