package io.intino.sumus.engine.builders.accumulators;

import io.intino.sumus.engine.Cube;

import java.time.LocalTime;
import java.util.Arrays;
import java.util.List;

import static io.intino.sumus.engine.Cube.indicator;

public class TimeAccumulator extends BuilderAccumulator {

	private final int[] count = new int[threadCount];
	private final int[] min = new int[threadCount];
	private final int[] max = new int[threadCount];

	public TimeAccumulator(String name) {
		super(name);
		Arrays.fill(this.min, Integer.MAX_VALUE);
		Arrays.fill(this.max, Integer.MIN_VALUE);
	}

	@Override
	public void add(Object value) {
		if (value == null) return;
		final int t = threadIndex();
		this.count[t]++;
		this.min[t] = Math.min(this.min[t], (int) value);
		this.max[t] = Math.max(this.max[t], (int) value);
	}

	@Override
	public List<Cube.Indicator> indicators() {
		int count = count();
		int min = min();
		int max = max();

		return List.of(
				indicator("count(" + name + ")", count),
				indicator("min(" + name + ")", minTime(min)),
				indicator("max(" + name + ")", maxTime(max))
		);
	}

	public int max() {
		return Arrays.stream(this.max).max().orElse(Integer.MIN_VALUE);
	}

	public int min() {
		return Arrays.stream(this.min).min().orElse(Integer.MAX_VALUE);
	}

	public int count() {
		return Arrays.stream(this.count).sum();
	}

	private LocalTime minTime(int min) {
		return min == Integer.MAX_VALUE ? LocalTime.MAX : LocalTime.ofSecondOfDay(min);
	}

	private LocalTime maxTime(int max) {
		return max == Integer.MIN_VALUE ? LocalTime.MIN : LocalTime.ofSecondOfDay(max);
	}
}
