package io.intino.sumus.engine.builders.accumulators;

import io.intino.sumus.engine.Cube;

import java.util.Arrays;
import java.util.List;

import static io.intino.sumus.engine.Cube.indicator;

public class IntegerAccumulator extends BuilderAccumulator {

	private final int[] count = new int[threadCount];
	private final long[] sum = new long[threadCount];
	private final long[] totalSum = new long[threadCount];
	private final long[] min = new long[threadCount];
	private final long[] max = new long[threadCount];

	public IntegerAccumulator(String name) {
		super(name);
		Arrays.fill(this.min, Long.MAX_VALUE);
		Arrays.fill(this.max, Long.MIN_VALUE);
	}

	@Override
	public void addTotal(Object value) {
		final int t = threadIndex();
		total[t]++;
		if(value != null) totalSum[t] += (long) value;
	}

	@Override
	public void add(Object value) {
		if (value == null) return;
		final int t = threadIndex();
		final long v = (long) value;
		this.count[t]++;
		this.sum[t] += v;
		this.min[t] = Math.min(this.min[t], v);
		this.max[t] = Math.max(this.max[t], v);
	}

	@Override
	public List<Cube.Indicator> indicators() {
		int count = count();
		long totalSum = totalSum();
		long sum = sum();
		long min = min();
		long max = max();

		return List.of(
				indicator("count(" + name + ")", count),
				indicator("ratio(" + name + ")", totalSum > 0 ? (double) sum / totalSum : 0),
				indicator("sum(" + name + ")", sum),
				indicator("average(" + name + ")", count > 0 ? (double) sum / count : 0),
				indicator("min(" + name + ")", min),
				indicator("max(" + name + ")", max)
		);
	}

	public long max() {
		return Arrays.stream(this.max).max().orElse(Long.MIN_VALUE);
	}

	public long min() {
		return Arrays.stream(this.min).min().orElse(Long.MAX_VALUE);
	}

	public long sum() {
		return Arrays.stream(this.sum).sum();
	}

	public long totalSum() {
		return Arrays.stream(this.totalSum).sum();
	}

	public int count() {
		return Arrays.stream(this.count).sum();
	}
}
