package io.intino.sumus.engine.builders;

import io.intino.sumus.engine.*;
import io.intino.sumus.engine.builders.accumulators.*;
import io.intino.sumus.engine.concurrency.SumusThreadFactory;
import io.intino.sumus.engine.filters.CompositeFilter;
import io.intino.sumus.engine.filters.SliceFilter;
import io.intino.sumus.engine.helpers.IgnoreCaseMap;
import io.intino.sumus.model.LedgerDefinition;

import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;

public class CubeBuilder {

	private final Ledger ledger;
	private final LedgerDefinition definition;
	private final Filter filter;
	private final List<Dimension> dimensions;

	public CubeBuilder(Ledger ledger, Filter filter, Dimension... dimensions) {
		this(ledger, filter, asList(dimensions));
	}

	public CubeBuilder(Ledger ledger, Filter filter, List<Dimension> dimensions) {
		this.ledger = ledger;
		this.definition = ledger.definition();
		this.filter = filter;
		this.dimensions = dimensions;
	}

	public Cube build() {
		synchronized (CubeBuilder.class) {
			return cubeOf(executeQuery());
		}
	}

	private CellBuilder[] executeQuery() {
		return filter == Filter.All ? new CellBuilder[0] : fill(withAccumulators(builders()));
	}

	private CellBuilder[] withAccumulators(CellBuilder[] builders) {
		for (CellBuilder builder : builders) createAccumulators(builder);
		return builders;
	}

	private CellBuilder[] fill(CellBuilder[] builders) {
		return SumusEngine.threadCount() > 1 ? fillMultithreading(builders) : fillSingleThread(builders);
	}

	private void fill(CellBuilder[] builders, Fact fact) {
		for (CellBuilder builder : builders) builder.add(fact);
	}

	private CellBuilder[] fillSingleThread(CellBuilder[] builders) {
		ExecutorService threadPool = Executors.newSingleThreadExecutor(new SumusThreadFactory("CubeBuilder-T"));
		threadPool.execute(() -> {
			for (Fact fact : ledger.facts(filter)) fill(builders, fact);
		});
		waitFor(threadPool);
		return builders;
	}

	private CellBuilder[] fillMultithreading(CellBuilder[] builders) {
		ExecutorService threadPool = SumusEngine.createThreadPool();
		List<Throwable> errors = Collections.synchronizedList(new ArrayList<>());

		for(Fact fact : ledger.facts(filter)) {
			threadPool.execute(() -> {
				try {
					fill(builders, fact);
				} catch (Throwable e) {
					errors.add(e);
				}
			});
		}

		waitFor(threadPool);

		if(!errors.isEmpty()) rethrowFirst(errors);

		return builders;
	}

	private void waitFor(ExecutorService threadPool) {
		try {
			threadPool.shutdown();
			threadPool.awaitTermination(SumusEngine.TIMEOUT_AMOUNT.get(), SumusEngine.TIMEOUT_UNIT.get());
		} catch (Throwable e) {
			throw new CubeBuilderException(e);
		}
	}

	private static void rethrowFirst(List<Throwable> errors) {
		Throwable e = errors.get(0);
		throw new CubeBuilderException("Error while executing CubeBuilder: " + e.getMessage(), e);
	}


	private void createAccumulators(CellBuilder builder) {
		builder.setAccumulators(ledger.attributes().stream()
				.filter(Attribute::isUsedInIndicators)
				.map(this::accumulatorOf)
				.toArray(Accumulator[]::new));
	}

	private Accumulator accumulatorOf(Attribute attribute) {
		if (attribute.type().isInteger()) return new IntegerAccumulator(attribute.name());
		if (attribute.type().isReal()) return new DoubleAccumulator(attribute.name());
		if (attribute.type().isDate()) return new DateAccumulator(attribute.name());
		if (attribute.type().isTime()) return new TimeAccumulator(attribute.name());
		return new CountAccumulator(attribute.name());
	}

	private Cube cubeOf(CellBuilder[] builders) {
		return new CubeImpl(builders);
	}

	public CellBuilder[] builders() {
		final int size = dimensions.size();
		List<Slice> slices = emptyList();
		List<CellBuilder> result = new ArrayList<>(size * size);
		for (int i = 0; i < size; i++)
			result.addAll(builders(slices, dimensions.subList(i, size)));
		result.add(new CellBuilder(definition.indicators));
		return result.toArray(CellBuilder[]::new);
	}

	private List<CellBuilder> builders(List<Slice> slices, List<Dimension> dimensions) {
		List<CellBuilder> result = export(slices);
		if (dimensions.isEmpty()) return result;
		List<? extends Slice> head = filter.crop(dimensions.get(0).slices());
		List<Dimension> tail = dimensions.subList(1, dimensions.size());
		for (Slice slice : head) {
			List<CellBuilder> cells = builders(join(slices, slice), tail);
			result.addAll(cells);
		}
		return result;
	}

	private List<CellBuilder> export(List<Slice> slices) {
		List<CellBuilder> result = slices.isEmpty() ? List.of() : List.of(new CellBuilder(definition.indicators, slices));
		return new ArrayList<>(result);
	}

	private List<Slice> join(List<Slice> slices,Slice slice) {
		List<Slice> result = new ArrayList<>(slices);
		result.add(slice);
		return result;
	}

	public class CubeImpl implements Cube {

		public final Map<String, Cell> cells;
		public final CellBuilder[] builders;

		public CubeImpl(CellBuilder[] builders) {
			this.builders = builders;
			this.cells = buildCells(builders);
		}

		@Override
		public List<? extends Dimension> dimensions() {
			return dimensions;
		}

		public List<? extends Cell> cells() {
			return new ArrayList<>(cells.values());
		}

		@Override
		public Map<String, ? extends Cell> cellMap() {
			return cells;
		}

		@Override
		public Iterable<Fact> facts(Filter filter) {
			Filter local = CompositeFilter.of(CubeBuilder.this.filter, filter);
			return local == Filter.All ? emptyList() : ledger.facts(local);
		}

		private Map<String, Cube.Cell> buildCells(CellBuilder[] builders) {
			return Arrays.stream(builders)
					.map(b -> b.cell(facts(CompositeFilter.of(filter, SliceFilter.of(b.slices())))))
					.collect(Collectors.toMap(Object::toString, c -> c, (a, b) -> b, IgnoreCaseMap::new));
		}
	}

	public static class CubeBuilderException extends RuntimeException {

		public CubeBuilderException() {
		}

		public CubeBuilderException(String message) {
			super(message);
		}

		public CubeBuilderException(String message, Throwable cause) {
			super(message, cause);
		}

		public CubeBuilderException(Throwable cause) {
			super(cause);
		}
	}
}
