package io.intino.sumus.engine.builders;

import io.intino.sumus.engine.Cube;
import io.intino.sumus.engine.Fact;
import io.intino.sumus.engine.Filter;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.engine.filters.SliceFilter;
import io.intino.sumus.model.CompositeIndicatorDefinition;
import io.intino.sumus.model.IndicatorDefinition;
import io.intino.sumus.model.SimpleIndicatorDefinition;

import java.util.*;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toMap;

public class CellBuilder {

    private final List<IndicatorDefinition> indicators;
    private final Filter filter;
    private final List<Slice> slices;
    private Accumulator[] accumulators = new Accumulator[0];

    public CellBuilder(List<IndicatorDefinition> indicators) {
        this.indicators = indicators;
        this.filter = Filter.None;
        this.slices = emptyList();
    }

    public CellBuilder(List<IndicatorDefinition> indicators, List<Slice> slices) {
        this.indicators = indicators;
        this.filter = SliceFilter.of(slices);
        this.slices = slices;
    }

    public void setAccumulators(Accumulator[] accumulators) {
        this.accumulators = accumulators;
    }

    public void add(Fact fact) {
        boolean isAccepted = filter.accepts(fact.idx());
        for (Accumulator accumulator : accumulators) {
            Object value = fact.value(accumulator.name());
            accumulator.addTotal(value);
            if (isAccepted) accumulator.add(value);
        }
    }

    @Override
    public String toString() {
        return slices.stream().map(Slice::name).collect(joining("-"));
    }

    public Cube.Cell cell(Iterable<Fact> facts) {
        return new CellImpl(facts);
    }

    public List<Slice> slices() {
        return slices;
    }

    public List<IndicatorDefinition> indicators() {
        return indicators;
    }

    public Filter filter() {
        return filter;
    }

    public Accumulator[] accumulators() {
        return accumulators;
    }

    public class CellImpl implements Cube.Cell {

        private Map<String, Cube.Indicator> indicatorValueMap;
        private final Iterable<Fact> facts;

        public CellImpl(Iterable<Fact> facts) {
            this.facts = facts;
        }

        public List<Accumulator> accumulators() {
            return Arrays.asList(accumulators);
        }

        @Override
        public List<? extends Slice> slices() {
            return slices;
        }

        @Override
        public List<? extends Cube.Indicator> indicators() {
            return new ArrayList<>(indicatorsMap().values());
        }

        @Override
        public Map<String, ? extends Cube.Indicator> indicatorsMap() {
            if (indicatorValueMap == null) indicatorValueMap = translate(calculations());
            return indicatorValueMap;
        }

        private Map<String, Cube.Indicator> translate(Map<String, Cube.Indicator> calculations) {
            Map<String, Cube.Indicator> map = new LinkedHashMap<>();
            indicators.forEach(i -> {
                Cube.Indicator indicator = i.isComposite() ?
                        indicator(map, i.asComposite()) :
                        indicator(calculations, i.asSimple());
                if (indicator == null) return;
                map.put(i.name(), indicator);
            });
            return map;
        }

        private Cube.Indicator indicator(Map<String, Cube.Indicator> calculations, SimpleIndicatorDefinition d) {
            return calculations.getOrDefault(d.label(), Cube.Indicator.None).as(d.name(), d.unit(), ratio(d.scale()));
        }

        private Cube.Indicator indicator(Map<String, Cube.Indicator> map, CompositeIndicatorDefinition c) {
            Cube.Indicator i1 = map.get(c.indicator1());
            Cube.Indicator i2 = map.get(c.indicator2());
            Object value = CompositeCalculator.calculate(c, i1, i2);
            if (value == null) return null;
            return Cube.indicator(c.name(), value).as(c.name(), c.unit(), ratio(c.scale()));
        }

        private double ratio(SimpleIndicatorDefinition.Scale scale) {
            return scale == null ? 1.0 : scale.ratio();
        }

        private Map<String, Cube.Indicator> calculations() {
            return Arrays.stream(accumulators)
                    .flatMap(a -> a.indicators().stream())
                    .collect(toMap(Cube.Indicator::name, i -> i, (a, b) -> a));
        }

        @Override
        public Iterable<Fact> facts() {
            return facts;
        }

        @Override
        public String toString() {
            return slices.stream().map(Slice::toString).collect(joining("-"));
        }
    }

}
