package io.intino.sumus.engine.builders;

import io.intino.sumus.engine.Cube;
import io.intino.sumus.engine.Fact;
import io.intino.sumus.engine.Filter;
import io.intino.sumus.engine.Slice;
import io.intino.sumus.engine.filters.SliceFilter;
import io.intino.sumus.model.IndicatorDefinition;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.*;

public class CellBuilder {

	private final List<IndicatorDefinition> indicators;
	private final Filter filter;
	private final List<Slice> slices;
	private Accumulator[] accumulators = new Accumulator[0];

	public CellBuilder(List<IndicatorDefinition> indicators) {
		this.indicators = indicators;
		this.filter = Filter.None;
		this.slices = emptyList();
	}

	public CellBuilder(List<IndicatorDefinition> indicators, List<Slice> slices) {
		this.indicators = indicators;
		this.filter = SliceFilter.of(slices);
		this.slices = slices;
	}

	public void setAccumulators(Accumulator[] accumulators) {
		this.accumulators = accumulators;
	}

	public void add(Fact fact) {
		boolean isAccepted = filter.accepts(fact.idx());
		for (Accumulator accumulator : accumulators) {
			Object value = fact.value(accumulator.name());
			accumulator.addTotal(value);
			if (isAccepted) accumulator.add(value);
		}
	}

	@Override
	public String toString() {
		return slices.stream().map(Slice::name).collect(joining("-"));
	}

	public Cube.Cell cell(Iterable<Fact> facts) {
		return new CellImpl(facts);
	}

	public List<Slice> slices() {
		return slices;
	}

	public List<IndicatorDefinition> indicators() {
		return indicators;
	}

	public Filter filter() {
		return filter;
	}

	public Accumulator[] accumulators() {
		return accumulators;
	}

	public class CellImpl implements Cube.Cell {

		private Map<String, Cube.Indicator> indicatorValueMap;
		private final Iterable<Fact> facts;

		public CellImpl(Iterable<Fact> facts) {
			this.facts = facts;
		}

		public List<Accumulator> accumulators() {
			return Arrays.asList(accumulators);
		}

		@Override
		public List<? extends Slice> slices() {
			return slices;
		}

		@Override
		public List<? extends Cube.Indicator> indicators() {
			return new ArrayList<>(indicatorsMap().values());
		}

		@Override
		public Map<String, ? extends Cube.Indicator> indicatorsMap() {
			if (indicatorValueMap == null) indicatorValueMap = translate(calculations());
			return indicatorValueMap;
		}

		private Map<String, Cube.Indicator> translate(Map<String, Cube.Indicator> calculations) {
			return indicators.stream()
					.map(d -> indicator(calculations, d))
					.collect(toMap(Cube.Indicator::name, i -> i));
		}

		private Cube.Indicator indicator(Map<String, Cube.Indicator> calculations, IndicatorDefinition d) {
			return calculations.getOrDefault(d.label(), Cube.Indicator.None).as(d.name(), d.unit(), ratio(d.scale()));
		}

		private double ratio(IndicatorDefinition.Scale scale) {
			return scale == null ? 1.0 : scale.ratio();
		}

		private Map<String, Cube.Indicator> calculations() {
			return Arrays.stream(accumulators)
					.flatMap(a -> a.indicators().stream())
					.collect(toMap(Cube.Indicator::name, i -> i, (a, b) -> a));
		}

		@Override
		public Iterable<Fact> facts() {
			return facts;
		}

		@Override
		public String toString() {
			return slices.stream()
					.map(Slice::toString)
					.collect(joining("-"));
		}
	}

}
