package io.intino.sumus.engine.builders;

import io.intino.sumus.engine.*;
import io.intino.sumus.engine.builders.accumulators.*;
import io.intino.sumus.engine.filters.CompositeFilter;
import io.intino.sumus.engine.filters.SliceFilter;
import io.intino.sumus.engine.helpers.IgnoreCaseMap;
import io.intino.sumus.model.LedgerDefinition;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;

public class CubeBuilder {
	private final Ledger ledger;
	private final LedgerDefinition definition;
	private final Filter filter;
	private final List<Dimension> dimensions;

	public CubeBuilder(Ledger ledger, Filter filter, Dimension... dimensions) {
		this(ledger, filter, asList(dimensions));
	}

	public CubeBuilder(Ledger ledger, Filter filter, List<Dimension> dimensions) {
		this.ledger = ledger;
		this.definition = ledger.definition();
		this.filter = filter;
		this.dimensions = dimensions;
	}

	public List<CellBuilder> builders() {
		List<CellBuilder> result = new ArrayList<>();
		int size = dimensions.size();
		for (int i = 0; i < size; i++)
			result.addAll(builders(List.of(), dimensions.subList(i, size)));
		result.add(new CellBuilder(definition.indicators));
		return result;
	}

	private List<CellBuilder> builders(List<Slice> slices, List<Dimension> dimensions) {
		List<CellBuilder> result = export(slices);
		if (dimensions.isEmpty()) return result;
		List<Slice> head = filter.crop(dimensions.get(0).slices());
		List<Dimension> tail = dimensions.subList(1, dimensions.size());
		for (Slice slice : head) {
			List<CellBuilder> cells = builders(join(slices, slice), tail);
			result.addAll(cells);
		}
		return result;
	}

	private List<CellBuilder> export(List<Slice> slices) {
		List<CellBuilder> result = slices.isEmpty() ? List.of() : List.of(new CellBuilder(definition.indicators, slices));
		return new ArrayList<>(result);
	}

	private List<Slice> join(List<Slice> slices,Slice slice) {
		List<Slice> result = new ArrayList<>(slices);
		result.add(slice);
		return result;
	}

	public Cube build() {
		return cubeOf(executeQuery());
	}

	private List<CellBuilder> executeQuery() {
		return filter == Filter.All ? emptyList() : fill(withAccumulators(builders()));
	}

	private List<CellBuilder> withAccumulators(List<CellBuilder> builders) {
		for (CellBuilder builder : builders) createAccumulators(builder);
		return builders;
	}

	private List<CellBuilder> fill(List<CellBuilder> builders) {
		for (Fact fact : ledger.facts(filter))
			if (filter.accepts(fact.idx())) fill(builders, fact);
		return builders;
	}

	private void fill(List<CellBuilder> builders, Fact fact) {
		for (CellBuilder builder : builders) builder.add(fact);
	}

	@SuppressWarnings("ResultOfMethodCallIgnored")
	private void terminate(ExecutorService executor) {
		try {
			executor.shutdown();
			executor.awaitTermination(40, TimeUnit.SECONDS);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
	}

	private void createAccumulators(CellBuilder builder) {
		List<Attribute> attributes = ledger.attributes();
		attributes.stream()
				.filter(Attribute::isUsedInIndicators)
				.forEach(a->builder.add(accumulatorOf(a)));
	}

	private BuilderAccumulator accumulatorOf(Attribute attribute) {
		if (attribute.type().isInteger()) return new IntegerAccumulator(attribute.name());
		if (attribute.type().isReal()) return new NumberAccumulator(attribute.name());
		if (attribute.type().isDate()) return new DateAccumulator(attribute.name());
		if (attribute.type().isTime()) return new TimeAccumulator(attribute.name());
		return new CountAccumulator(attribute.name());
	}

	private Cube cubeOf(List<CellBuilder> builders) {
		return new Cube() {
			private final Map<String, Cell> cells = buildCells();

			@Override
			public List<Dimension> dimensions() {
				return dimensions;
			}

			public List<Cell> cells() {
				return new ArrayList<>(cells.values());
			}

			@Override
			public Map<String, Cell> cellMap() {
				return cells;
			}

			@Override
			public Iterable<Fact> facts(Filter filter) {
				Filter local = CompositeFilter.of(CubeBuilder.this.filter, filter);
				return local == Filter.All ? emptyList() : ledger.facts(local);
			}

			private Map<String, Cube.Cell> buildCells() {
				return builders.stream()
						.map(b -> b.cell(facts(CompositeFilter.of(filter, SliceFilter.of(b.slices)))))
						.collect(Collectors.toMap(Object::toString, c -> c, (a, b) -> b, IgnoreCaseMap::new));
			}

		};
	}


}
