package io.intino.sumus.chronos.util;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;

public class ChannelsHelper {

	public static SeekableByteChannel makeSeekable(ReadableByteChannel channel) {
		return new ReadableByteChannelWithPosition(channel);
	}

	public static SeekableByteChannel newSeekableByteChannel(byte[] bytes) {
		return new ByteArraySeekableByteChannel(bytes);
	}

	private static class ReadableByteChannelWithPosition implements SeekableByteChannel {

		private final ReadableByteChannel channel;
		private long position;

		public ReadableByteChannelWithPosition(ReadableByteChannel channel) {
			this.channel = channel;
		}

		@Override
		public int read(ByteBuffer dst) throws IOException {
			int read = channel.read(dst);
			position += read;
			return read;
		}

		@Override
		public int write(ByteBuffer src) {
			throw new UnsupportedOperationException();
		}

		@Override
		public long position() {
			return position;
		}

		@Override
		public java.nio.channels.SeekableByteChannel position(long newPosition) {
			throw new UnsupportedOperationException();
		}

		@Override
		public long size() {
			throw new UnsupportedOperationException();
		}

		@Override
		public java.nio.channels.SeekableByteChannel truncate(long size) {
			throw new UnsupportedOperationException();
		}

		@Override
		public boolean isOpen() {
			return channel.isOpen();
		}

		@Override
		public void close() throws IOException {
			channel.close();
		}
	}

	private static class ByteArraySeekableByteChannel implements SeekableByteChannel {

		private final ByteBuffer content;

		public ByteArraySeekableByteChannel(byte[] content) {
			this.content = ByteBuffer.wrap(content);
		}

		@Override
		public int read(ByteBuffer dst) throws IOException {
			int oldPos = content.position();
			content.limit(Math.min(oldPos + dst.remaining(), content.capacity()));
			dst.put(content);
			content.limit(content.capacity());
			return content.position() - oldPos;
		}

		@Override
		public int write(ByteBuffer src) throws IOException {
			throw new UnsupportedOperationException();
		}

		@Override
		public long position() throws IOException {
			return content.position();
		}

		@Override
		public SeekableByteChannel position(long newPosition) throws IOException {
			content.position((int) newPosition);
			return this;
		}

		@Override
		public long size() throws IOException {
			return content.capacity();
		}

		@Override
		public SeekableByteChannel truncate(long size) throws IOException {
			throw new UnsupportedEncodingException();
		}

		@Override
		public boolean isOpen() {
			return true;
		}

		@Override
		public void close() throws IOException {
		}
	}
}
