package io.intino.sumus.chronos.timelines.blocks;

import io.intino.sumus.chronos.TimelineStore;
import io.intino.sumus.chronos.Period;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Objects;

public class TimeModel implements TimelineStore.TimeModel {

	public static final int SIZE = 2 + 4 + 8; // mark + instant + period

	private final Instant instant;
	private final Period period;

	public TimeModel(Instant instant, Period period) {
		this.instant = instant;
		this.period = period;
	}

	@Override
	public Instant instant() {
		return instant;
	}

	@Override
	public Period period() {
		return period;
	}

	@Override
	public ChronoUnit unit() {
		return period.unit;
	}

	@Override
	public long duration() {
		return period.duration();
	}

	@Override
	public Instant next(Instant instant) {
		return period.next(instant);
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;
		TimelineStore.TimeModel timeModel = (TimelineStore.TimeModel) o;
		return Objects.equals(instant, timeModel.instant()) && Objects.equals(period, timeModel.period());
	}

	@Override
	public int hashCode() {
		return Objects.hash(instant, period);
	}

	@Override
	public String toString() {
		return "from " + instant + " each " + period;
	}

	public static ByteBuffer serialize(TimelineStore.TimeModel timeModel) {
		return serialize(timeModel, ByteBuffer.allocate(SIZE)).clear();
	}

	public static ByteBuffer serialize(TimelineStore.TimeModel timeModel, ByteBuffer buffer) {
		buffer.putShort(MARK);
		buffer.putLong(timeModel.instant().toEpochMilli());
		buffer.putShort((short) timeModel.period().amount);
		buffer.putShort((short) timeModel.period().unit.ordinal());
		return buffer;
	}

	public static TimelineStore.TimeModel deserialize(ReadableByteChannel channel, boolean readMark) throws IOException {
		ByteBuffer buffer = ByteBuffer.allocate(readMark ? SIZE : SIZE - Short.BYTES);
		channel.read(buffer);
		buffer.position(readMark ? Short.BYTES : 0);

		Instant instant = Instant.ofEpochMilli(buffer.getLong());
		short periodAmount = buffer.getShort();
		short periodUnit = buffer.getShort();

		return new TimeModel(instant, new Period(periodAmount, ChronoUnit.values()[periodUnit]));
	}
}
