package io.intino.sumus.chronos.timelines.blocks;

import io.intino.sumus.chronos.TimelineStore;
import io.intino.sumus.chronos.Magnitude;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.joining;

public class SensorModel implements TimelineStore.SensorModel {

	static final int HEADER_SIZE = 2 + 4; // mark + size
	static final String MAGNITUDE_SEPARATOR = "\n";

	private final long position;
	private final Magnitude[] magnitudes;
	private final Map<Magnitude, Integer> index;

	public SensorModel(String... magnitudes) {
		this(-1, magnitudes);
	}

	public SensorModel(long position, String... magnitudes) {
		this(stream(magnitudes).map(Magnitude::new).toArray(Magnitude[]::new));
	}

	public SensorModel(Magnitude... magnitudes) {
		this(-1, magnitudes);
	}

	public SensorModel(long position, Magnitude... magnitudes) {
		this.position = position;
		this.magnitudes = magnitudes;
		this.index = new HashMap<>(magnitudes.length);
		initIndex(magnitudes);
	}

	public long position() {
		return position;
	}

	public Magnitude[] magnitudes() {
		return magnitudes;
	}

	public Magnitude get(int i) {
		return i >= 0 ? magnitudes[i] : null;
	}

	public Magnitude get(String measurement) {
		return get(indexOf(new Magnitude(measurement)));
	}

	public int indexOf(Magnitude magnitude) {
		return index.getOrDefault(magnitude,-1);
	}

	@Override
	public int indexOf(String magnitude) {
		return index.getOrDefault(new Magnitude(magnitude),-1);
	}

	public int size() {
		return magnitudes.length;
	}

	public boolean has(String measurement) {
		return index.containsKey(new Magnitude(measurement));
	}

	public boolean has(Magnitude magnitude) {
		return index.containsKey(magnitude);
	}

	private void initIndex(Magnitude[] magnitudes) {
		for (int i = 0; i < magnitudes.length; i++)
			index.put(magnitudes[i], i);
	}

	@Override
	public boolean equals(Object o) {
		if (o == null || getClass() != o.getClass()) return false;
		return this == o || Arrays.equals(magnitudes, ((TimelineStore.SensorModel) o).magnitudes());
	}

	@Override
	public int hashCode() {
		return Arrays.hashCode(magnitudes);
	}

	@Override
	public String toString() {
		return stream(magnitudes).map(Magnitude::toString).collect(joining("\n"));
	}


	public static ByteBuffer serialize(TimelineStore.SensorModel sensorModel) {
		byte[] utf8 = String.join(MAGNITUDE_SEPARATOR, sensorModel.magnitudes()).getBytes(UTF_8);
		return serialize(utf8, ByteBuffer.allocate(HEADER_SIZE + utf8.length)).clear();
	}

	public static ByteBuffer serialize(byte[] utf8, ByteBuffer buffer) {
		buffer.putShort(MARK);
		buffer.putInt(utf8.length);
		buffer.put(utf8);
		return buffer;
	}

	public static TimelineStore.SensorModel deserialize(ReadableByteChannel channel, boolean readMark) throws IOException {
		long position = channel instanceof SeekableByteChannel ? ((SeekableByteChannel)channel).position() : -1;
		if(!readMark && position > 0) position -= Short.BYTES;

		ByteBuffer headerBuffer = ByteBuffer.allocate(readMark ? HEADER_SIZE : HEADER_SIZE - Short.BYTES);
		channel.read(headerBuffer);

		ByteBuffer utf8 = ByteBuffer.allocate(headerBuffer.getInt(readMark ? Short.BYTES : 0));
		channel.read(utf8);
		String[] magnitudes = new String(utf8.array(), UTF_8).split(MAGNITUDE_SEPARATOR, -1);

		return new SensorModel(position, magnitudes);
	}
}
