package io.intino.sumus.chronos.timelines.blocks;

import io.intino.sumus.chronos.TimelineStore;

import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Iterator;
import java.util.stream.IntStream;

public class Data implements TimelineStore.Data {

	public static final int HEADER_SIZE = 2 + 4; // mark + size
	public static final int MEASUREMENT_BYTE_SIZE = 8;

	private final long position;
	private final int recordByteSize;
	private final Instant[] instants;
	private final ByteBuffer buffer;

	public Data(long position, Instant[] instants, int numMeasurementsPerRecord, ByteBuffer buffer) {
		this.position = position;
		this.instants = instants;
		this.recordByteSize = numMeasurementsPerRecord * MEASUREMENT_BYTE_SIZE;
		this.buffer = buffer;
	}

	public long position() {
		return position;
	}

	@Override
	public int numRecords() {
		return buffer.remaining() / recordByteSize;
	}

	@Override
	public int numMeasurementsPerRecord() {
		return recordByteSize / MEASUREMENT_BYTE_SIZE;
	}

	@Override
	public int recordByteSize() {
		return recordByteSize;
	}

	@Override
	public ByteBuffer dataBuffer() {
		return buffer;
	}

	@Override
	public TimelineStore.Data.Record record(int index) {
		return new Record(index);
	}

	@Override
	public Iterator<TimelineStore.Data.Record> iterator() {
		return IntStream.range(0, numRecords())
				.<TimelineStore.Data.Record>mapToObj(Record::new)
				.iterator();
	}

	public static int calculateRecordByteSize(int numMeasurements) {
		return numMeasurements * MEASUREMENT_BYTE_SIZE;
	}

	public class Record implements TimelineStore.Data.Record {

		private final int index;
		private final int offset;

		private Record(int index) {
			this.index = index;
			this.offset = index * recordByteSize;
		}

		@Override
		public int index() {
			return index;
		}

		@Override
		public Instant instant() {
			return instants[index];
		}

		@Override
		public double[] values() {
			double[] values = new double[numMeasurements()];
			for(int i = 0; i < values.length; i++) values[i] = get(i);
			return values;
		}

		@Override
		public double get(int i) {
			if(i < 0 || i >= numMeasurements()) throw new IndexOutOfBoundsException("Index must be >= 0 && < " + numMeasurements());
			return buffer.getDouble(offset + i * MEASUREMENT_BYTE_SIZE);
		}

		@Override
		public int byteSize() {
			return recordByteSize();
		}

		@Override
		public int numMeasurements() {
			return numMeasurementsPerRecord();
		}
	}
}
