package io.intino.sumus.chronos.timelines;

import io.intino.sumus.chronos.Magnitude;
import io.intino.sumus.chronos.MeasurementsVector;
import io.intino.sumus.chronos.Period;
import io.intino.sumus.chronos.TimelineStore;
import io.intino.sumus.chronos.timelines.blocks.Data;
import io.intino.sumus.chronos.timelines.blocks.Header;
import io.intino.sumus.chronos.timelines.blocks.SensorModel;
import io.intino.sumus.chronos.timelines.blocks.TimeModel;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.DoubleBuffer;
import java.nio.channels.SeekableByteChannel;
import java.time.Instant;
import java.util.Collection;

import static io.intino.sumus.chronos.timelines.blocks.Data.MEASUREMENT_BYTE_SIZE;

public class TimelineWriter implements AutoCloseable {

	private static final int DEFAULT_BUFFER_SIZE = 32 * 1024;

	// I/O
	private final SeekableByteChannel channel;
	// Buffering
	private ByteBuffer buffer;
	private int dataBlockPosition;
	// Chronos
	private final Header header;
	private TimelineStore.SensorModel sensorModel;
	private TimelineStore.TimeModel timeModel;

	public TimelineWriter(String sensor, SeekableByteChannel channel) throws IOException {
		this(sensor, channel, DEFAULT_BUFFER_SIZE);
	}

	public TimelineWriter(String sensor, SeekableByteChannel channel, int bufferSize) throws IOException {
		this.channel = channel;
		this.header = new Header(sensor);
		this.buffer = ByteBuffer.allocate(Math.max(bufferSize, 1024));
		reserveOrReadHeader();
	}

	public Header header() {
		return header;
	}

	public TimelineStore.SensorModel sensorModel() {
		return sensorModel;
	}

	public TimelineStore.TimeModel timeModel() {
		return timeModel;
	}

	public Instant threshold() {
		return timeModel == null ? Instant.EPOCH : timeModel.next(header.next());
	}

	private void reserveOrReadHeader() throws IOException {
		if(channel.position() == 0) reserveHeader(); else readHeader();
	}

	public TimelineWriter sensorModel(TimelineStore.SensorModel sensorModel) throws IOException {
		if(this.sensorModel != null && this.sensorModel.contains(sensorModel)) return this;
		this.sensorModel = sensorModel;
		writeSensorModel();
		return this;
	}

	public TimelineWriter sensorModel(String... magnitudes) throws IOException {
		return sensorModel(new SensorModel(magnitudes));
	}

	public TimelineWriter sensorModel(Magnitude... magnitudes) throws IOException {
		return sensorModel(new SensorModel(magnitudes));
	}

	public TimelineWriter timeModel(TimelineStore.TimeModel timeModel) throws IOException {
		if(timeModel.equals(this.timeModel)) return this;
		this.timeModel = timeModel;
		writeTimeModel();
		return this;
	}

	public TimelineWriter timeModel(Instant instant, Period period) throws IOException {
		return timeModel(new TimeModel(instant, period));
	}

	public TimelineWriter set(Instant instant) throws IOException {
		if (instant.isBefore(threshold())) return this;
		return timeModel(new TimeModel(instant, timeModel.period()));
	}

	public TimelineWriter set(MeasurementsVector measurementsVector) throws IOException {
		return set(measurementsVector.toArray());
	}

	public TimelineWriter set(double[] values) throws IOException {
		return set(values, 0, values.length);
	}

	public TimelineWriter set(double[] values, int offset, int length) throws IOException {
		ensureCapacityOrFlush(length * MEASUREMENT_BYTE_SIZE);
		beginDataBlock();
		for(int i = 0;i < length;i++) buffer.putDouble(values[offset + i]);
		header.step(timeModel);
		return this;
	}

	public TimelineWriter set(Collection<Number> values) throws IOException {
		ensureCapacityOrFlush(values.size() * MEASUREMENT_BYTE_SIZE);
		beginDataBlock();
		for(Number n : values) buffer.putDouble(n.doubleValue());
		header.step(timeModel);
		return this;
	}

	public TimelineWriter set(ByteBuffer values) throws IOException {
		ensureCapacityOrFlush(values.remaining());
		beginDataBlock();
		put(values);
		header.step(timeModel);
		return this;
	}

	public TimelineWriter set(DoubleBuffer values) throws IOException {
		int numBytes = values.remaining() * Double.BYTES;
		ensureCapacityOrFlush(numBytes);
		beginDataBlock();
		buffer.asDoubleBuffer().put(values);
		buffer.position(buffer.position() + numBytes);
		header.step(timeModel);
		return this;
	}

	private void ensureCapacityOrFlush(int bytes) throws IOException {
		if(dataBlockPosition <= 0) bytes += Data.HEADER_SIZE;
		if(bytes > buffer.remaining())
			flush();
	}

	private void beginDataBlock() {
		if(dataBlockPosition > 0) return;
		buffer.putShort(Data.MARK);
		buffer.putInt(0);
		dataBlockPosition = buffer.position();
	}

	private void endDataBlock() {
		if(dataBlockPosition <= 0) return;
		buffer.putInt(dataBlockPosition - Integer.BYTES, buffer.position() - dataBlockPosition);
		dataBlockPosition = 0;
	}

	public void flush() throws IOException {
		if (buffer == null || buffer.position() == 0) return;
		endDataBlock();
		commit(buffer.flip());
	}

	@Override
	public void close() throws IOException {
		if(!channel.isOpen()) return;
		endDataBlock();
		flush();
		writeHeader();
		channel.close();
		buffer = null;
	}

	private void writeHeader() throws IOException {
		channel.position(0);
		commit(Header.serialize(header));
		channel.position(channel.size());
	}

	private void writeTimeModel() throws IOException {
		header.setTimeModel(currentPosition(), timeModel);
		endDataBlock();
		ByteBuffer timeModelBytes = TimeModel.serialize(timeModel);
		ensureCapacityOrFlush(timeModelBytes.remaining());
		put(timeModelBytes);
	}

	private void writeSensorModel() throws IOException {
		header.setSensorModel(currentPosition());
		endDataBlock();
		ByteBuffer sensorModelBytes = SensorModel.serialize(sensorModel);
		ensureCapacityOrFlush(sensorModelBytes.remaining());
		put(sensorModelBytes);
	}

	private void commit(ByteBuffer buffer) throws IOException {
		channel.write(buffer);
		buffer.clear();
	}

	private void reserveHeader() throws IOException {
		channel.position(Header.SIZE);
	}

	private void readHeader() throws IOException {
		channel.position(0);
		TimelineReader.readHeader(channel, header);
		this.sensorModel = TimelineReader.readSensorModel(header.sensorModelPosition, channel);
		this.timeModel = TimelineReader.readTimeModel(header.timeModelPosition, channel);
		channel.position(channel.size());
	}

	private long currentPosition() throws IOException {
		return channel.position() + buffer.position();
	}

	private void put(ByteBuffer values) throws IOException {
		final int originalLimit = values.limit();
		while(values.hasRemaining()) {
			values.limit(Math.min(values.position() + buffer.limit(), originalLimit));
			ensureCapacityOrFlush(values.remaining());
			buffer.put(values);
			values.limit(originalLimit);
		}
	}

	private static String[] magnitudes() {
		String[] m = new String[1000];
		for(int i = 0;i < m.length;i++) {
			m[i] = i + "E81SEI048:lamp.osram.xbo.3000.dhp.l." + i;
		}
		return m;
	}
}
