package io.intino.sumus.chronos.timelines;

import io.intino.sumus.chronos.TimelineStore;
import io.intino.sumus.chronos.InvalidChronosBlockMarkException;
import io.intino.sumus.chronos.timelines.blocks.Data;
import io.intino.sumus.chronos.timelines.blocks.Header;
import io.intino.sumus.chronos.timelines.blocks.SensorModel;
import io.intino.sumus.chronos.timelines.blocks.TimeModel;
import io.intino.sumus.chronos.util.ChannelsHelper;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.time.Instant;
import java.util.Iterator;

public class TimelineReader implements Iterator<TimelineStore.Block>, AutoCloseable {

	private final SeekableByteChannel channel;
	private final ByteBuffer markAndSizeBuffer;
	private final TimelineStore.Header header;
	private TimelineStore.TimeModel timeModel;
	private TimelineStore.SensorModel sensorModel;
	private TimelineStore.Block current;
	private Instant currentInstant;

	public TimelineReader(ReadableByteChannel channel) throws IOException {
		this.channel = ChannelsHelper.makeSeekable(channel);
		this.markAndSizeBuffer = ByteBuffer.allocate(Integer.BYTES);
		this.header = readHeader(channel, new Header());
		this.current = readNextBlock();
	}

	public long position() throws IOException {
		return channel.position();
	}

	public TimelineStore.Header header() {
		return header;
	}

	public TimelineStore.TimeModel timeModel() {
		return timeModel;
	}

	public TimelineStore.SensorModel sensorModel() {
		return sensorModel;
	}

	@Override
	public boolean hasNext() {
		return current != null;
	}

	@Override
	public TimelineStore.Block next() {
		TimelineStore.Block next = current;
		current = readNextBlock();
		return next;
	}

	@Override
	public void close() throws IOException {
		channel.close();
	}

	private TimelineStore.Block readNextBlock() {
		try {
			int read = channel.read(markAndSizeBuffer.position(0).limit(Short.BYTES));
			if(read != Short.BYTES) return null;
			return readBlock(markAndSizeBuffer.getShort(0));
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	private TimelineStore.Block readBlock(short mark) throws IOException {
		switch(mark) {
			case TimelineStore.TimeModel.MARK: return readTimeModel();
			case TimelineStore.SensorModel.MARK: return readSensorModel();
			case TimelineStore.Data.MARK: return readData();
		}
		throw new InvalidChronosBlockMarkException("Invalid Chronos block mark: 0x" + Integer.toHexString(mark).toUpperCase());
	}

	private TimelineStore.Block readTimeModel() throws IOException {
		timeModel = TimeModel.deserialize(channel, false);
		currentInstant = timeModel.instant();
		return timeModel;
	}

	private TimelineStore.Block readSensorModel() throws IOException {
		return sensorModel = SensorModel.deserialize(channel, false);
	}

	private TimelineStore.Block readData() throws IOException {
		long position = position();
		if(position > Short.BYTES) position -= Short.BYTES;

		channel.read(markAndSizeBuffer.position(0).limit(Integer.BYTES));

		ByteBuffer data = ByteBuffer.allocate(markAndSizeBuffer.getInt(0));
		channel.read(data);

		return readDataRecordsFrom(position, data.clear());
	}

	private TimelineStore.Block readDataRecordsFrom(long position, ByteBuffer data) {
		int recordSize = Data.calculateRecordByteSize(sensorModel.size());
		int numRecords = data.remaining() / recordSize;

		Instant[] instants = new Instant[numRecords];
		for(int i = 0;i < numRecords;i++) {
			instants[i] = currentInstant;
			currentInstant = timeModel.next(currentInstant);
		}

		return new Data(position, instants, sensorModel.size(), data);
	}

	public static Header readHeader(ReadableByteChannel channel, Header header) throws IOException {
		ByteBuffer buffer = ByteBuffer.allocate(Header.SIZE);
		int read = channel.read(buffer);
		if(read < Header.SIZE) throw new IllegalStateException("Could not read Chronos Header. Bytes read = " + read);
		Header.deserialize(buffer.clear(), header);
		return header;
	}

	public static TimelineStore.Header readHeader(java.nio.channels.SeekableByteChannel channel) throws IOException {
		return TimelineReader.readHeader(channel, new Header());
	}

	public static TimelineStore.SensorModel readSensorModel(long sensorModelPosition, java.nio.channels.SeekableByteChannel channel) throws IOException {
		channel.position(sensorModelPosition);
		return SensorModel.deserialize(channel, true);
	}

	public static TimelineStore.TimeModel readTimeModel(long timeModelPosition, java.nio.channels.SeekableByteChannel channel) throws IOException {
		channel.position(timeModelPosition);
		return TimeModel.deserialize(channel, true);
	}
}
