package io.intino.sumus.chronos.processors;

import io.intino.sumus.chronos.Magnitude;
import io.intino.sumus.chronos.Timeline;
import io.intino.sumus.chronos.TimelineImpl;

import java.util.HashMap;
import java.util.Map;

import static java.lang.Double.NaN;
import static java.lang.Double.isNaN;

public class Interpolator {
	private final Timeline timeline;

	public Interpolator(Timeline timeline) {
		this.timeline = timeline;
	}

	public Timeline execute() {
		return new TimelineImpl.Builder(timeline.instants()).put(valuesOf(timeline)).build();
	}

	private Map<Magnitude, double[]> valuesOf(Timeline timeline) {
		Map<Magnitude, double[]> result = new HashMap<>();
		for (Magnitude magnitude : timeline.magnitudes())
			result.put(magnitude, new Operator(timeline.get(magnitude).values).execute());
		return result;
	}

	public static class Operator {
		private final double[] values;

		public Operator(double[] values) {
			this.values = values;
		}

		public double[] execute() {
			double[] result = new double[values.length];
			for (int i = 0; i < result.length; i++)
				result[i] = isNumber(values[i]) ? values[i] : interpolate(i);
			return result;
		}

		private double interpolate(int i) {
			int left = firstLeftNumber(i - 1);
			int right = firstRightNumber(i + 1);
			return left >= 0 && right < values.length ? interpolate(left, i, right) : NaN;
		}

		private double interpolate(int left, int i, int right) {
			return values[left] + f(i - left, right - left, values[right] - values[left]);
		}

		private int firstLeftNumber(int i) {
			while (i >= 0 && isNaN(values[i])) i--;
			return i;
		}

		private int firstRightNumber(int i) {
			while (i < values.length && isNaN(values[i])) i++;
			return i;
		}

		private static boolean isNumber(double result) {
			return !isNaN(result);
		}

		private double f(double d, double dx, double dy) {
			double c = dy / dx;
			double a = 2 * (c - dy / dx) / (dx * dx);
			double b = (3 * dy / dx - 2 * c - c) / dx;
			return a * d * d * d + b * d * d + c * d;
		}
	}
}
