package io.intino.sumus.chronos.models.descriptive.timeseries;

import io.intino.sumus.chronos.*;
import io.intino.sumus.chronos.models.descriptive.sequence.Sequence.Quantization;
import io.intino.sumus.chronos.models.descriptive.timeline.Summary;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.Arrays;

import static java.lang.Double.NaN;
import static java.lang.Integer.parseInt;
import static java.lang.Math.*;
import static java.time.ZoneOffset.UTC;

public class Trend {
	public static final Trend Null = new Trend(Timeline.Null, 0);
	public final Timeline timeline;
	public final int window;

	public static Builder of(TimeSeries timeSeries, int window) {
		return new Builder(timeSeries, window);
	}

	private Trend(Timeline timeline, int window) {
		this.timeline = timeline;
		this.window = window;
	}

	public State at(Instant instant) {
		return at(timeline.at(instant));
	}

	private State at(TimelineImpl.Point point) {
		return new State(point);
	}

	public State last() {
		return at(timeline.last());
	}

	public class State {
		public final TimelineImpl.Point point;
		public final Direction direction;
		public final Strength strength;
		public final Volatility volatility;
		public final Probability probability;

		public class Probability {
			public final double volatility;
			public final double value;

			public Probability(TimelineImpl.Point point) {
				this.value = min(evaluate(point, "open"), evaluate(point, "close"));
				this.volatility = evaluate(point, "ATR");
			}

			private double evaluate(TimelineImpl.Point point, String magnitude) {
				return point != null ?
						timeline.get(magnitude).probabilityOf(point.value(magnitude)) :
						NaN;
			}
		}

		public State(TimelineImpl.Point point) {
			this.point = point;
			this.direction = direction(point);
			this.strength = strength(point);
			this.volatility = volatility(point);
			this.probability = new Probability(point);
		}

		private Direction direction(TimelineImpl.Point point) {
			return point != null ? Direction.of(point.value("RSI")) : Direction.Neutral;
		}

		private Strength strength(TimelineImpl.Point point) {
			return point != null ? Strength.of(point.value("ADX")) : Strength.Weak;
		}

		private Volatility volatility(TimelineImpl.Point point) {
			return point != null ? Volatility.of(point.value("ATR")) : Volatility.VeryLow;
		}

		public boolean isStrong() {
			return strength.ordinal() >= Strength.Strong.ordinal();
		}

		@Override
		public String toString() {
			return point + ": " + direction + "," + strength + "," + volatility;
		}
	}


	public static class Builder {
		private final TimeSeries timeSeries;
		private final double sd;
		private final int window;

		private Builder(TimeSeries timeSeries, int window) {
			this.timeSeries = timeSeries;
			this.sd = timeSeries.distribution().sd();
			this.window = window;
		}

		public Trend by(Period period) {
			return sd != 0 ? new Trend(analyzeBy(period), window) : Trend.Null;
		}

		private Timeline analyzeBy(Period period) {
			Summary summary = summaryWith(period);
			return createTimelineFrom(summary);
		}

		private Summary summaryWith(Period period) {
			return Summary.of(asTimeline(timeSeries)).by(Quantization.of(period));
		}

		private static Timeline asTimeline(TimeSeries series) {
			return Timeline.builder(series.instants).put(new Magnitude("value", series.model), series).build();
		}

		private Timeline createTimelineFrom(Summary summary) {
			Timeline.Builder builder = Timeline.builder(summary.length());
			Distribution yesterday = summary.first().distribution("value");
			for (Summary.Point point : summary) {
				Distribution today = point.distribution("value");
				builder.set(parseInstant(point.label()));
				builder.set("open", today.open);
				builder.set("max", today.max);
				builder.set("min", today.min);
				builder.set("close", today.close);
				builder.set("sum", today.sum);
				builder.set("mean", today.mean());
				builder.set("TR", trueRange(today, yesterday));
				builder.set("DM+", dmMax(today, yesterday));
				builder.set("DM-", dmMin(today, yesterday));
				builder.set("RSI", relativeStrengthIndex(point));
				yesterday = today;
			}
			return terminate(builder.build());
		}

		private Instant parseInstant(String label) {
			int[] date = decomposeDate(label);
			return LocalDateTime.of(date[0], date[1], date[2], date[3], date[4], date[5]).toInstant(UTC);
		}

		private int[] decomposeDate(String label) {
			int[] date = new int[6];
			date[0] = parseInt(label.substring(0, 4));
			date[1] = 1;
			date[2] = 2;
			for (int i = 1, pos = 4; i < 6 && pos < label.length(); i++, pos += 2)
				date[i] = parseInt(label.substring(pos, pos + 2));
			return date;
		}

		private double relativeStrengthIndex(Summary.Point today) {
			double[] values = today.backward().limit(window + 1).mapToDouble(p -> p.distribution("value").close).toArray();
			for (int i = values.length - 1; i >= 1; i--)
				values[i] = values[i] - values[i - 1];
			double averageGain = Arrays.stream(values).skip(1).filter(v -> v > 0).average().orElse(0);
			double averageLoss = Arrays.stream(values).skip(1).filter(v -> v < 0).average().orElse(0);
			return averageLoss != 0 ? 100 - 100 / (1 + averageGain / abs(averageLoss)) : 100;
		}

		private Timeline terminate(Timeline timeline) {
			if (timeline.instantsCount() < window) return Timeline.Null;
			TimeSeries atr = timeline.get("TR").movingAverage(window);
			TimeSeries factor = atr.inverse().times(100);
			TimeSeries highDI = timeline.get("DM+").exponentialMovingAverage(window).times(factor);
			TimeSeries lowDI = timeline.get("DM-").exponentialMovingAverage(window).times(factor);
			TimeSeries directionalIndex = highDI.minus(lowDI).abs().dividedBy(highDI.plus(lowDI)).times(100);
			return timeline
					.add("ATR:tail=up", atr.times(1. / sd))
					.add("ADX", directionalIndex.exponentialMovingAverage(window));
		}

		private static double dmMax(Distribution today, Distribution yesterday) {
			double change = today.close - yesterday.close;
			double highChange = today.max - yesterday.max;
			return change > 0 && change > highChange ? change : 0;
		}

		private static double dmMin(Distribution today, Distribution yesterday) {
			double change = yesterday.close - today.close;
			double lowChange = today.min - yesterday.min;
			return change > 0 && change > lowChange ? change : 0;
		}

		private static double trueRange(Distribution today, Distribution yesterday) {
			return max(today.max, yesterday.close) - min(today.min, yesterday.close);
		}

	}

	public enum Direction {
		Down, Neutral, Up;

		public static Direction of(double rsi) {
			return rsi > 70 ? Down : rsi < 30 ? Up : Neutral;
		}

	}

	public enum Strength {
		Weak, Medium, Strong, VeryStrong;

		public static Strength of(double adx) {
			if (adx < 25) return Weak;
			if (adx < 50) return Medium;
			if (adx < 75) return Strong;
			return VeryStrong;
		}

	}

	public enum Volatility {
		VeryLow, Low, Medium, High, VeryHigh;

		public static Volatility of(double atr) {
			if (atr < 0.5) return VeryLow;
			if (atr < 1.0) return Low;
			if (atr < 1.5) return Medium;
			if (atr < 2.0) return High;
			return VeryHigh;
		}

	}
}
