package io.intino.sumus.chronos.itl;

import io.intino.sumus.chronos.TimelineStore;
import io.intino.sumus.chronos.Period;
import io.intino.sumus.chronos.timelines.TimelineReader;

import java.io.*;
import java.nio.channels.Channels;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;

import static java.lang.Double.isNaN;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.joining;

public class ItlExporter {

	private static final DecimalFormat TheDecimalFormatter = formatter();

	public static void export(String id, File src, File dst) throws IOException {
		export(id, new FileInputStream(src), new FileOutputStream(dst));
	}

	public static void export(String id, InputStream src, OutputStream dst) throws IOException {
		Period[] lastPeriod = new Period[1];
		try(BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(dst))) {
			writer.write("@id " + id + "\n");
			try(TimelineReader reader = new TimelineReader(Channels.newChannel(src))) {
				while(reader.hasNext()) {
					TimelineStore.Block block = reader.next();
					if (block instanceof TimelineStore.TimeModel) write((TimelineStore.TimeModel) block, writer, lastPeriod);
					else if (block instanceof TimelineStore.SensorModel) write((TimelineStore.SensorModel) block, writer);
					else if (block instanceof TimelineStore.Data) write((TimelineStore.Data) block, writer);
				}
			}
		}
	}

	private static void write(TimelineStore.SensorModel data, Writer writer) throws IOException {
		writer.write("@measurements " + String.join(",", data.magnitudes()) + "\n");
	}

	private static void write(TimelineStore.TimeModel timeModel, Writer writer, Period[] lastPeriod) throws IOException {
		writer.write("@instant " + timeModel.instant().toString() + "\n");
		if (timeModel.period().equals(lastPeriod[0])) return;
		writer.write("@period " + timeModel.period() + "\n");
		lastPeriod[0] = timeModel.period();
	}

	private static void write(TimelineStore.Data data, Writer writer) throws IOException {
		for(TimelineStore.Data.Record record : data) writer.write(toString(record.values()) + "\n");
	}

	private static String toString(double[] values) {
		return stream(values).mapToObj(ItlExporter::toString).collect(joining("\t")).stripTrailing();
	}

	private static String toString(double value) {
		return isNaN(value) ? "" : format(value);
	}

	private static String format(double value) {
		return isNaN(value) ? "" : TheDecimalFormatter.format(value);
	}

	private static DecimalFormat formatter() {
		DecimalFormat df = new DecimalFormat("#.#############");
		df.setDecimalFormatSymbols(new DecimalFormatSymbols(Locale.ENGLISH));
		return df;
	}
}
