package io.intino.sumus.chronos;

import io.intino.sumus.chronos.processors.Interpolator;
import io.intino.sumus.chronos.processors.Resampler;

import java.time.Instant;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Stream;

import static io.intino.sumus.chronos.Magnitude.NaN;
import static java.lang.Double.NaN;
import static java.lang.Math.*;
import static java.util.Arrays.binarySearch;
import static java.util.Arrays.copyOfRange;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.IntStream.iterate;
import static java.util.stream.IntStream.range;

public class TimelineImpl implements Timeline {

	private final Instant[] instants;
	private final Map<Magnitude, double[]> measurements;

	public TimelineImpl(Instant[] instants, Map<Magnitude, double[]> measurements) {
		this.instants = instants;
		this.measurements = measurements;
	}

	@Override
	public int magnitudesCount() {
		return measurements.size();
	}

	@Override
	public int instantsCount() {
		return instants.length;
	}

	@Override
	public Set<Magnitude> magnitudes() {
		return measurements.keySet();
	}

	@Override
	public Instant[] instants() {
		return instants;
	}

	@Override
	public Map<Magnitude, double[]> measurements() {
		return measurements;
	}

	@Override
	public Instant instant(int index) {
		return instants[index];
	}

	@Override
	public boolean has(String magnitude) {
		return has(new Magnitude(magnitude));
	}

	@Override
	public boolean has(Magnitude magnitude) {
		return measurements.containsKey(magnitude);
	}

	@Override
	public TimeSeries get(String magnitude) {
		return has(magnitude) ? get(find(magnitude)) : emptyTimeSeries();
	}

	@Override
	public TimeSeries get(Magnitude magnitude) {
		return new TimeSeries(magnitude.model, instants, measurements.getOrDefault(magnitude, NaN(instants.length)));
	}

	private TimeSeries emptyTimeSeries() {
		return new TimeSeries(Magnitude.Model.Default, instants, NaN(instants.length));
	}

	@Override
	public boolean isBefore(Timeline timeline) {
		if (timeline.instantsCount() == 0) return false;
		return isEndOfTimes(timeline.instant(0));
	}

	@Override
	public boolean isAfter(Timeline timeline) {
		if (timeline.instantsCount() == 0) return false;
		return isBeginningOfTimes(timeline.instant(0));
	}

	@Override
	public Point first() {
		return point(0);
	}

	@Override
	public Point last() {
		return point(lastIndex());
	}

	@Override
	public Point at(Instant instant) {
		return point(indexOf(instant));
	}

	@Override
	public Timeline head(int length) {
		return sub(0, limitHigh(length));
	}

	@Override
	public Timeline tail(int length) {
		return sub(limitLow(instantsCount() - length), instantsCount());
	}

	@Override
	public Timeline from(Instant instant) {
		if (isEndOfTimes(instant)) return Timeline.empty();
		return sub(indexOf(instant), instantsCount());
	}

	@Override
	public Timeline from(Instant instant, int length) {
		if (isEndOfTimes(instant)) return Timeline.empty();
		int from = indexOf(instant);
		return length > 0 ?
				sub(from, limitHigh(from + length)) :
				sub(limitLow(from + 1 + length), limitHigh(from + 1));
	}

	@Override
	public Timeline from(Instant instant, Instant to) {
		if (isEndOfTimes(instant)) return Timeline.empty();
		if (isEndOfTimes(to)) return from(instant);
		return sub(indexOf(instant), indexOf(to));
	}

	@Override
	public Timeline to(Instant instant) {
		if (isEndOfTimes(instant)) return this;
		return sub(0, indexOf(instant));
	}

	private Point point(int index) {
		return isInRange(index) ? new PointImpl(index) : null;
	}

	private boolean isBeginningOfTimes(Instant instant) {
		return firstInstant().isAfter(instant);
	}

	private boolean isEndOfTimes(Instant instant) {
		return lastInstant().isBefore(instant);
	}

	private int limitLow(int offset) {
		return max(0, offset);
	}

	private int limitHigh(int offset) {
		return min(offset, instantsCount());
	}

	private Instant firstInstant() {
		return instants[0];
	}

	private Instant lastInstant() {
		return instants[lastIndex()];
	}

	private Timeline sub(int from, int to) {
		return new TimelineImpl(instants(from, to), values(from, to));
	}

	private Magnitude find(String label) {
		return measurements.keySet().stream()
				.filter(m -> m.label.equalsIgnoreCase(label))
				.findFirst()
				.orElse(null);
	}

	@Override
	public Stream<Point> stream() {
		return iterate(0, i -> i < instantsCount(), i -> i + 1).mapToObj(this::point);
	}

	@Override
	public Iterator<Point> iterator() {
		return new Iterator<>() {
			int i = 0;

			@Override
			public boolean hasNext() {
				return i < instantsCount();
			}

			@Override
			public Point next() {
				return new PointImpl(i++);
			}
		};
	}

	@Override
	public Timeline add(Timeline timeline) {
		assert Arrays.equals(instants(), timeline.instants());
		TimelineImpl result = new TimelineImpl(instants, new HashMap<>(measurements));
		for (Magnitude magnitude : timeline.magnitudes()) {
			TimeSeries series = result.has(magnitude) ? timeline.get(magnitude).plus(get(magnitude)) : timeline.get(magnitude);
			result.put(magnitude, series);
		}
		return result;
	}

	@Override
	public Timeline add(String magnitude, TimeSeries timeSeries) {
		return add(new Magnitude(magnitude), timeSeries);
	}

	@Override
	public Timeline add(Magnitude magnitude, TimeSeries timeSeries) {
		assert Arrays.equals(timeSeries.instants, instants);
		TimelineImpl result = new TimelineImpl(instants, new HashMap<>(measurements));
		result.put(magnitude, timeSeries);
		return result;
	}

	@Override
	public Timeline add(String magnitude, Function<Timeline, TimeSeries> function) {
		return add(new Magnitude(magnitude), function);
	}

	@Override
	public Timeline add(Magnitude magnitude, Function<Timeline, TimeSeries> function) {
		TimelineImpl result = new TimelineImpl(instants, new HashMap<>(measurements));
		result.put(magnitude, function.apply(this));
		return result;
	}

	@Override
	public Timeline concat(Timeline timeline) {
		assert isBefore(timeline);
		TimelineImpl result = new TimelineImpl(TimeSeries.concat(instants(), timeline.instants()), new HashMap<>());
		Set<Magnitude> magnitudes = concat(magnitudes(), timeline.magnitudes());
		for (Magnitude magnitude : magnitudes)
			result.put(magnitude, TimeSeries.concat(get(magnitude).values, timeline.get(magnitude).values));
		return result;
	}

	@Override
	public Timeline compose(Function<Magnitude, Magnitude> function) {
		TimelineImpl result = new TimelineImpl(instants, new HashMap<>());
		for (Magnitude magnitude : magnitudes()) {
			Magnitude m = function.apply(magnitude);
			TimeSeries series = result.has(m) ? result.get(m).plus(get(magnitude)) : get(magnitude);
			result.put(m, series);
		}
		return result;
	}

	private void put(Magnitude magnitude, TimeSeries series) {
		measurements.put(magnitude, series.values);
	}

	private void put(Magnitude magnitude, double[] values) {
		put(magnitude, new TimeSeries(Magnitude.Model.Default, instants, values));
	}

	@Override
	public Timeline resampleBy(Period period) {
		return instantsCount() > 0 ? resampleBy(period, sizeWith(period)) : this;
	}

	@Override
	public Timeline resampleBy(Period period, int size) {
		return new Resampler(this).execute(period, size);
	}

	@Override
	public Timeline execute(Program program) {
		return program.run(this);
	}

	@Override
	public Timeline interpolate() {
		return new Interpolator(this).execute();
	}

	private static Set<Magnitude> concat(Set<Magnitude> a, Set<Magnitude> b) {
		Set<Magnitude> magnitudes = new HashSet<>();
		magnitudes.addAll(a);
		magnitudes.addAll(b);
		return magnitudes;
	}

	private int sizeWith(Period period) {
		return period.length(first().instant(), last().instant().plusSeconds(1));
	}

	@Override
	public boolean equals(Object o) {
		if (o == null || getClass() != o.getClass()) return false;
		return this == o || equals((TimelineImpl) o);
	}

	private boolean equals(TimelineImpl timeline) {
		return instantsAreEqual(timeline) && measurements.size() == timeline.magnitudesCount() && valuesAreEqual(timeline);
	}

	public boolean instantsAreEqual(TimelineImpl timeline) {
		return Arrays.equals(instants, timeline.instants);
	}

	public boolean valuesAreEqual(TimelineImpl timeline) {
		return measurements.keySet().stream()
				.map(m -> Arrays.equals(measurements.get(m), timeline.measurements.get(m)))
				.reduce(true, (a, b) -> a && b);
	}

	@Override
	public int hashCode() {
		int result = Objects.hash(measurements);
		result = 31 * result + Arrays.hashCode(instants);
		return result;
	}

	private int indexOf(Instant instant) {
		int index = binarySearch(instants, instant);
		return index >= 0 ? index : min(abs(-index - 1), lastIndex());
	}

	private int lastIndex() {
		return instantsCount() - 1;
	}

	private boolean isInRange(int index) {
		return index >= 0 && index < instantsCount();
	}

	private Instant[] instants(int from, int to) {
		return copyOfRange(this.instants, from, to);
	}

	private Map<Magnitude, double[]> values(int from, int to) {
		return magnitudes().stream()
				.collect(toMap(m -> m, m -> copyOfRange(valuesOf(m), from, to)));
	}

	private double[] valuesOf(Magnitude magnitude) {
		return measurements.get(magnitude);
	}

	public Map<String, double[]> collect(String... magnitudes) {
		return Arrays.stream(magnitudes)
				.collect(toMap(m -> m, m -> get(m).values, (a, b) -> b));
	}

	@Override
	public String toString() {
		return first() + ".. " + last();
	}

	public class PointImpl implements Point {
		final int index;

		public PointImpl(int index) {
			this.index = index;
		}

		@Override
		public Instant instant() {
			return instants[index];
		}

		@Override
		public Set<Magnitude> magnitudes() {
			return TimelineImpl.this.magnitudes();
		}

		@Override
		public boolean has(String magnitude) {
			return has(new Magnitude(magnitude));
		}

		@Override
		public boolean has(Magnitude magnitude) {
			return TimelineImpl.this.has(magnitude);
		}

		@Override
		public double value(String magnitude) {
			return value(new Magnitude(magnitude));
		}

		@Override
		public double value(Magnitude magnitude) {
			return has(magnitude) ? valuesOf(magnitude)[index] : NaN;
		}

		@Override
		public String toString() {
			return instant() + magnitudes().stream().map(m -> "\t" + format(value(m))).collect(joining());
		}

		@Override
		public Stream<Point> forward() {
			return range(index, instantsCount()).mapToObj(TimelineImpl.this::point);
		}

		@Override
		public Stream<Point> backward() {
			return iterate(index, i -> i >= 0, i -> i - 1).mapToObj(TimelineImpl.this::point);
		}

		private String format(double value) {
			long v = (long) value;
			return value == v ? String.valueOf(v) : String.valueOf(value);
		}

		@Override
		public Point next() {
			return step(1);
		}

		@Override
		public Point prev() {
			return step(-1);
		}

		@Override
		public Point step(int value) {
			return point(index + value);
		}

		@Override
		public boolean equals(Object o) {
			if (o == null || getClass() != o.getClass()) return false;
			return this == o || index == ((PointImpl) o).index;
		}

		@Override
		public int hashCode() {
			return Objects.hash(index);
		}
	}

	public static class Builder implements Timeline.Builder {

		private final Instant[] instants;
		private final Map<Magnitude, double[]> magnitudes;
		private int position;

		public Builder(int size) {
			this(new Instant[size]);
		}

		public Builder(Instant[] instants) {
			this.instants = instants;
			this.magnitudes = new HashMap<>();
			this.position = -1;
		}

		@Override
		public Magnitude get(String magnitude) {
			return magnitudes().stream()
					.filter(m -> m.label.equals(magnitude))
					.findFirst()
					.orElse(null);
		}

		private Set<Magnitude> magnitudes() {
			return magnitudes.keySet();
		}

		@Override
		public Timeline.Builder put(Timeline timeline) {
			this.magnitudes.putAll(timeline.measurements());
			return this;
		}

		@Override
		public Timeline.Builder put(Magnitude magnitude, TimeSeries series) {
			assert instants.length == series.length();
			this.magnitudes.put(magnitude, series.values);
			return this;
		}

		@Override
		public Timeline.Builder put(Magnitude magnitude, double[] values) {
			assert instants.length == values.length;
			this.magnitudes.put(magnitude, values);
			return this;
		}

		@Override
		public Timeline.Builder put(Map<Magnitude, double[]> values) {
			values.keySet().forEach(m -> put(m, new TimeSeries(Magnitude.Model.Default, instants, values.get(m))));
			return this;
		}

		@Override
		public Timeline.Builder set(Instant instant) {
			instants[++position] = instant;
			return this;
		}

		@Override
		public boolean isComplete() {
			return position + 1 >= instants.length;
		}

		@Override
		public void set(TimelineStore.SensorModel sensorModel, TimelineStore.Data dataBlock) {
			Magnitude[] magnitudes = sensorModel.magnitudes();
			for(Magnitude magnitude : magnitudes) register(magnitude);

			position = Math.max(this.position, 0);

			for (TimelineStore.Data.Record record : dataBlock) {
				final int position = this.position + record.index();
				instants[position] = record.instant();
				for (int i = 0; i < record.numMeasurements(); i++) {
					setMeasurement(position, magnitudes[i], record.get(i));
				}
			}

			position += dataBlock.numRecords();
		}

		@Override
		public void set(Magnitude[] magnitudes, Instant instant, double[] values) {
			set(instant);
			for(int i = 0;i < values.length;i++) setMeasurement(position, magnitudes[i], values[i]);
		}

		@Override
		public Timeline.Builder set(String magnitude, double value) {
			set(new Magnitude(magnitude), value);
			return this;
		}

		@Override
		public Timeline.Builder set(Magnitude magnitude, double value) {
			if (magnitude == null) return this;
			register(magnitude);
			setMeasurement(position, magnitude, value);
			return this;
		}

		private void setMeasurement(int position, Magnitude magnitude, double value) {
			if (!Double.isNaN(value)) magnitudes.get(magnitude)[position] = value;
		}

		@Override
		public void register(Magnitude magnitude) {
			if (magnitudes.containsKey(magnitude)) return;
			magnitudes.put(magnitude, NaN(instants.length));
		}

		@Override
		public Timeline build() {
			return new TimelineImpl(instants, magnitudes);
		}

		@Override
		public TimeSeries series(Magnitude magnitude) {
			return new TimeSeries(Magnitude.Model.Default, instants, magnitudes.get(magnitude));
		}
	}
}
