package io.intino.sumus.chronos;

import io.intino.alexandria.logger.Logger;

import java.io.*;
import java.time.Instant;
import java.util.*;

import static io.intino.sumus.chronos.State.*;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class ReelFile {
	private final File file;
	private final Header header;
	private final Instant from;

	private ReelFile(File file, Header header) {
		this.file = file;
		this.header = header;
		this.from = readFirstInstantFrom(file);
	}

	public static ReelFile open(File file) throws IOException {
		return new ReelFile(file, Header.read(file));
	}

	public static ReelFile create(File file) throws IOException {
		return new ReelFile(file, Header.create(file));
	}

	public Set<String> signals() {
		return header.signals();
	}

	public Set<String> signals(State state) {
		if (state == On) return header.onSignals();
		Set<String> signals = new HashSet<>(signals());
		signals.removeAll(header.onSignals());
		return signals;
	}

	public List<Group> groups() {
		return header.groups.subList(1, header.groups.size());
	}

	public Group group(String name) {
		return header.group(name);
	}

	public State lastStateOf(String signal) {
		return header.stateOf(signal);
	}

	public Instant lastUpdateOf(String signal) {
		return header.instantOf(signal);
	}

	public Shot lastShotOf(String signal) {
		return signals().contains(signal) ? new Shot(lastUpdateOf(signal), lastStateOf(signal), signal) : null;
	}

	public List<Shot> lastShots() {
		return signals().stream().map(this::lastShotOf).collect(toList());
	}

	public List<Shot> lastShots(String group) {
		return header.exists(group) ? lastShots(header.group(group)) : emptyList();
	}

	public List<Shot> lastShots(Group group) {
		return group.signals.stream().map(this::lastShotOf).collect(toList());
	}

	public Session session() {
		return new Session();
	}

	public class Session implements AutoCloseable {
		private BufferedOutputStream output;

		public Session() {
			try {
				output = new BufferedOutputStream(new FileOutputStream(file, true));
			} catch (FileNotFoundException e) {
				Logger.error(e);
			}
		}

		public Session set(Instant instant, String group, String... signals) throws IOException {
			return set(instant, header.create(group), signals);
		}

		public Session set(Instant instant, Group group, String... signals) throws IOException {
			List<Shot> shots = new ArrayList<>();
			shots.addAll(header.shotsToTurnOff(instant, group.signalsThatAreNotIn(signals)));
			shots.addAll(header.shotsToTurnOn(instant, Set.of(signals)));
			return append(group, shots);
		}

		public Session append(String group, Shot... shots) throws IOException {
			return append(group, List.of(shots));
		}

		public Session append(String group, List<Shot> shots) throws IOException {
			return append(header.create(group), shots);
		}

		private Session append(Group group, List<Shot> shots) throws IOException {
			header.add(group);
			for (Shot shot : shots) {
				if (header.stateOf(shot.signal) == shot.state) continue;
				output.write(serialize(shot));
				header.set(shot.signal, shot.state);
				header.set(shot.signal, shot.ts);
				group.put(shot.signal);
			}
			return this;
		}

		@Override
		public void close() throws IOException {
			output.flush();
			header.write(file);
		}
	}

	public Instant start() {
		return from;
	}


	private byte[] serialize(Shot shot) {
		try (ByteArrayOutputStream bos = new ByteArrayOutputStream();
			 DataOutputStream dos = new DataOutputStream(bos)) {
			dos.writeLong(shot.ts.toEpochMilli());
			dos.writeByte(shot.state == On ? 1 : 0);
			dos.writeInt(shot.signal.hashCode());
			return bos.toByteArray();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public ReelReader reel(Instant from, Instant to) {
		return new ReelReader(from, to);
	}

	public ReelReader reel(Instant to) {
		return new ReelReader(from, to);
	}

	public ReelReader reel() {
		return new ReelReader(from, Instant.now());
	}

	private Instant readFirstInstantFrom(File file) {
		try (DataInputStream is = new DataInputStream(new BufferedInputStream(new FileInputStream(file)))) {
			is.skip(Header.SIZE);
			long epoch = is.readLong();
			return Instant.ofEpochMilli(epoch);
		} catch (IOException e) {
			return Instant.now();
		}
	}

	public class ReelReader extends Reader {
		public ReelReader(Instant from, Instant to) {
			super(from, to);
		}

		public Reel by(Period period) {
			Reel.ReelBuilder builder = new Reel.ReelBuilder(from, to, header.groups()).by(period);
			execute(builder);
			return builder.close();
		}
	}

	public class Reader {
		protected final Instant from;
		protected final Instant to;

		public Reader(Instant from, Instant to) {
			this.from = from;
			this.to = to;
		}

		protected void execute(ShotCollector shotCollector) {
			try (DataInputStream is = new DataInputStream(new BufferedInputStream(new FileInputStream(file)))) {
				is.skipBytes(Header.SIZE);
				while (true) {
					long epoch = is.readLong();
					byte state = is.readByte();
					int signal = is.readInt();
					shotCollector.add(new Shot(Instant.ofEpochMilli(epoch), State.of(state==1), header.get(signal)));
				}
			} catch (IOException ignored) {

			}
		}
	}

	private static class Header {
		private static final int SIZE = 8192;
		private final Map<Integer, String> dictionary;
		private final Map<String, Instant> signals;
		private final List<Group> groups;
		private final Group baseGroup;

		public Header(List<Group> groups, Map<String,Instant> signals, Map<Integer, String> dictionary) {
			this.dictionary = dictionary;
			this.signals = signals;
			this.groups = new ArrayList<>(groups);
			this.baseGroup = groups.get(0);
		}

		public Set<String> signals() {
			return signals.keySet();
		}

		public Set<String> onSignals() {
			return baseGroup.signals;
		}

		public State stateOf(String signal) {
			return baseGroup.contains(signal) ? On : Off;
		}

		public Instant instantOf(String signal) {
			return signals.get(signal);
		}

		public void set(String signal, State state) {
			if (state == On)
				baseGroup.put(signal);
			else
				baseGroup.remove(signal);
		}

		public void set(String signal, Instant ts) {
			signals.put(signal, ts);
		}

		public void add(Group group) {
			if (exists(group)) return;
			groups.add(group);
		}

		private boolean exists(String group) {
			return groups.stream().anyMatch(g -> g.name.equals(group));
		}

		private boolean exists(Group group) {
			return groups.stream().anyMatch(g -> g.equals(group));
		}

		private Group group(String name) {
			return groups.stream().filter(g -> g.name.equals(name)).findFirst().orElse(null);
		}

		public Group create(String name) {
			Group group = group(name);
			if (group == null) groups.add(group = new Group(name));
			return group;
		}

		public static Header create(File file) throws IOException {
			try (RandomAccessFile raf = new RandomAccessFile(file, "rw")) {
				raf.write(new byte[SIZE]);
			}
			return new Header(List.of(new Group("~")), new HashMap<>(), new HashMap<>());
		}

		public static Header read(File file) {
			try (DataInputStream is = new DataInputStream(new FileInputStream(file))) {
				Map<Integer, String> dictionary = readDictionary(is);
				Map<String, Instant> signals = readSignals(is, dictionary);
				List<Group> groups = readGroups(is,dictionary);
				return new Header(groups, signals, dictionary);
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
		}

		private static Map<Integer,String> readDictionary(DataInputStream is) throws IOException {
			Map<Integer, String> result = new HashMap<>();
			while (true) {
				String label = is.readUTF();
				if (label.isEmpty()) break;
				result.put(label.hashCode(), label);
			}
			return result;
		}

		private static Map<String,Instant> readSignals(DataInputStream is, Map<Integer, String> dictionary) throws IOException {
			Map<String, Instant> result = new HashMap<>();
			while (true) {
				int id = is.readInt();
				if (id == 0) break;
				result.put(dictionary.get(id), Instant.ofEpochMilli(is.readLong()));
			}
			return result;
		}

		private static List<Group> readGroups(DataInputStream is, Map<Integer, String> dictionary) throws IOException {
			List<Group> list = new ArrayList<>();
			while (true) {
				int id = is.readInt();
				if (id == 0) break;
				Group group = read(dictionary.get(id), is, dictionary);
				list.add(group);
			}
			return list;
		}

		private static Group read(String name, DataInputStream is, Map<Integer, String> dictionary) throws IOException {
			Set<String> set = new HashSet<>();
			while (true) {
				int signal = is.readInt();
				if (signal == 0) break;
				set.add(dictionary.get(signal));
			}
			return new Group(name, set);
		}

		public void write(File file) throws IOException {
			try (RandomAccessFile raf = new RandomAccessFile(file, "rw")) {
				raf.seek(0);
				writeDictionary(raf);
				writeSignals(raf);
				writeGroups(raf);
			}
		}

		private void writeDictionary(RandomAccessFile raf) throws IOException {
			updateDictionary();
			for (String word : dictionary.values()) raf.writeUTF(word);
			raf.writeUTF("");
		}

		private void updateDictionary() {
			for (Group group : groups) put(group.name);
			for (String signal : signals.keySet()) put(signal);
		}

		private void writeSignals(RandomAccessFile raf) throws IOException {
			for (String signal : signals.keySet()) {
				raf.writeInt(signal.hashCode());
				raf.writeLong(signals.get(signal).toEpochMilli());
			}
			raf.writeInt(0);
		}

		private void put(String name) {
			if (dictionary.containsKey(name.hashCode())) return;
			dictionary.put(name.hashCode(), name);
		}

		private void writeGroups(RandomAccessFile raf) throws IOException {
			for (Group group : groups) writeGroup(raf, group);
			raf.writeInt(0);
		}

		private static void writeGroup(RandomAccessFile raf, Group group) throws IOException {
			raf.writeInt(group.name.hashCode());
			for (String signal : group.signals)
				raf.writeInt(signal.hashCode());
			raf.writeInt(0);
		}

		public List<Group> groups() {
			return groups;
		}


		public List<Shot> shotsToTurnOn(Instant now, Set<String> signals) {
			return signals.stream()
					.filter(s -> !s.isEmpty() && stateOf(s) == Off)
					.map(s -> Shot.on(now, s))
					.collect(toList());
		}

		public List<Shot> shotsToTurnOff(Instant instant, Set<String> signals) {
			return signals.stream()
					.filter(s -> !s.isEmpty() && stateOf(s) == On)
					.map(s -> Shot.off(instant, s))
					.collect(toList());
		}

		private String get(int signal) {
			return dictionary.get(signal);
		}

	}


}
