package io.intino.sumus.chronos;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static java.lang.Integer.toBinaryString;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.IntStream.range;

public class Reel {
	public final Instant from;
	public final Instant to;
	public final int step;
	public final int length;
	private final Map<String, byte[]> signals;
	private final List<Group> groups;

	private Reel(Instant from, Instant to, int step, int length, Map<String, byte[]> signals, List<Group> groups) {
		this.from = from;
		this.to = to;
		this.step = step;
		this.length = length;
		this.signals = signals;
		this.groups = groups;
	}

	public String get(String signal) {
		return new Line(signal).serialize();
	}

	public List<String> signals() {
		return new ArrayList<>(signals.keySet());
	}

	public List<String> signals(String group) {
		return signalsOf(find(group));
	}

	private Group find(String group) {
		return groups.stream().filter(g -> g.name.equals(group)).findFirst().orElse(new Group(group));
	}

	private List<String> signalsOf(Group group) {
		return group != null ? signals.keySet().stream().filter(group::contains).collect(toList()) : emptyList();
	}

	private class Line {
		final String signal;
		final byte[] bytes;
		final boolean last;

		Line(String signal) {
			this.signal = signal;
			this.bytes = signals.getOrDefault(signal, new byte[0]);
			this.last = get(length-1);
		}

		boolean get(int position) {
			int index = position / 8;
			return get(index < bytes.length ? bytes[index] : 0, position % 8);
		}

		boolean get(byte chunk, int bit) {
			return ((chunk >> (7-bit)) & 1) == 1;
		}

		private String serialize() {
			StringBuilder sb = new StringBuilder(length);
			for (byte b : bytes) sb.append(chunk(b));
			while (sb.length() < length) sb.append(' ');
			return sb.substring(0, length);
		}

	}

	public static class ReelBuilder implements ShotCollector {
		private final Instant from;
		private final Instant to;
		private final Map<Integer, String> names;
		private final Map<Integer, byte[]> signals;
		private final List<Group> groups;
		private int step;

		public ReelBuilder(Instant from, Instant to) {
			this(from, to, emptyList());
		}

		public ReelBuilder(Instant from, Instant to, List<Group> groups) {
			this.from = from;
			this.to = to;
			this.step = 1;
			this.names = new HashMap<>();
			this.signals = new HashMap<>();
			this.groups = groups;
		}

		public void add(Shot shot) {
			if (shot.state == State.On)
				on(shot.signal, shot.ts);
			else
				off(shot.signal, shot.ts);
		}

		public ReelBuilder on(String signal, Instant ts) {
			createSignalNameIfNotExists(signal.hashCode(), signal);
			return on(signal.hashCode(), ts);
		}

		private ReelBuilder on(int hash, Instant ts) {
			if (isInRange(ts)) fillFrom(indexOf(ts), true, signalOf(hash));
			return this;
		}

		public ReelBuilder off(String signal, Instant ts) {
			return off(signal.hashCode(), ts);
		}

		private ReelBuilder off(int hash, Instant ts) {
			if (isInRange(ts)) fillFrom(indexOf(ts), false, signalOf(hash));
			return this;
		}

		private boolean isInRange(Instant ts) {
			return ts.getEpochSecond() <= to.getEpochSecond();
		}

		private byte[] signalOf(int hash) {
			createIfNotExists(hash);
			return signals.get(hash);
		}

		private void createIfNotExists(int hash) {
			if (signals.containsKey(hash)) return;
			signals.put(hash, new byte[size()]);
		}

		private void createSignalNameIfNotExists(int hash, String signal) {
			if (names.containsKey(hash)) return;
			names.put(hash, signal);
		}

		private int size() {
			return (int) Math.ceil(length() / 8.);
		}

		private int length() {
			return (int) indexOf(to);
		}

		private void fillFrom(long index, boolean set, byte[] values) {
			int position = (int) (index >> 3);
			if (position < 0 || position >= values.length) return;
			values[position] = set ?
					(byte) (values[position] | mask(offset(index))) :
					(byte) (values[position] & ~mask(offset(index)));
			byte value = (byte) (set ? 0xFF : 0);

			for (int i = position + 1, size = size(); i < size; i++) values[i] = value;
		}

		private static byte offset(long index) {
			return (byte) (index & 7);
		}

		private static byte mask(int offset) {
			return (byte) ((1 << (8 - offset)) - 1);
		}

		private long indexOf(Instant ts) {
			long index = (ts.getEpochSecond() - from.getEpochSecond()) / step;
			return index >= 0 ? index : 0;
		}

		public Reel close() {
			return new Reel(from, to, step, length(), consolidateSignals(), groups);
		}

		private Map<String, byte[]> consolidateSignals() {
			return signals.keySet().stream()
					.filter(this::hasAnyNonZero)
					.collect(toMap(names::get, signals::get));
		}

		private boolean hasAnyNonZero(int hash) {
			return hasAnyNonZero(signals.get(hash));
		}

		private static boolean hasAnyNonZero(byte[] bytes) {
			return range(0, bytes.length).anyMatch(i -> bytes[i] != 0);
		}

		public ReelBuilder by(Period period) {
			this.step = (int) period.duration();
			return this;
		}

		public ReelBuilder by(int seconds) {
			this.step = seconds;
			return this;
		}
	}


	private static final Map<Integer, String> Chunks = createChunks();

	private static String chunk(int i) {
		return Chunks.get(i & 0xFF);
	}

	private static Map<Integer, String> createChunks() {
		return range(0, 256).boxed()
				.collect(toMap(i -> i, Reel::createChunk));
	}

	private static String createChunk(int i) {
		return toBinaryString((i & 0xFF) + 0x100)
				.substring(1)
				.replace('0', ' ')
				.replace('1', '-');
	}

}
