package io.intino.sumus.chronos.models.descriptive.sequence;

import io.intino.sumus.chronos.Period;
import io.intino.sumus.chronos.Timeline;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static java.util.stream.IntStream.iterate;

public class Sequence implements Iterable<Sequence.Point> {
	private final String[] symbols;
	private final int[] tokens;
	private final Map<String, Integer> index;

	public static Builder of(Timeline timeline) {
		return new Builder(timeline);
	}

	public static Sequence of(String... tokens) {
		return new Sequence(tokens);
	}

	private Sequence(String[] tokens) {
		this.symbols = Arrays.stream(tokens).distinct().toArray(String[]::new);
		this.index = index(symbols);
		this.tokens = Arrays.stream(tokens).mapToInt(this::indexOf).toArray();
	}

	private static Map<String, Integer> index(String[] symbols) {
		Map<String, Integer> index = new HashMap<>();
		for (String symbol : symbols)
			index.put(symbol, index.size());
		return index;
	}

	public Histogram histogram() {
		return Histogram.of(this);
	}

	public String[] symbols() {
		return symbols;
	}

	public int length() {
		return tokens.length;
	}

	public Point first() {
		return point(0);
	}

	public Point last() {
		return point(length()-1);
	}

	public Point point(int index) {
		return index >= 0 && index < length()? new Point(index) : null;
	}

	public Stream<Point> stream() {
		return IntStream.range(0,length())
				.mapToObj(this::point);
	}


	public class Point {
		private final int index;

		public Point(int index) {
			this.index = index;
		}

		public int token() {
			return tokens[index];
		}

		public String symbol() {
			return symbols[token()];
		}

		public Point next() {
			return point(index+1);
		}

		public Point prev() {
			return point(index-1);
		}

		public Point step(int offset) {
			return point(index+offset);
		}

		public Stream<Point> forward() {
			return IntStream.range(index,length())
					.mapToObj(Sequence.this::point);
		}

		public Stream<Point> backward() {
			return iterate(index, i -> i >= 0, i -> i - 1)
					.mapToObj(Sequence.this::point);
		}

		@Override
		public String toString() {
			return symbol();
		}
	}

	public String symbol(int index) {
		return symbols[tokens[index]];
	}

	public int indexOf(String symbol) {
		return index.getOrDefault(symbol, -1);
	}

	@Override
	public Iterator<Point> iterator() {
		return new Iterator<>() {
			int index = 0;
			@Override
			public boolean hasNext() {
				return index < tokens.length;
			}

			@Override
			public Point next() {
				return point(index++);
			}
		};
	}


	public static class Builder {
		private final Timeline timeline;

		public Builder(Timeline timeline) {
			this.timeline = timeline;
		}

		public Sequence by(Period period) {
			return by(Quantization.of(period));
		}

		public Sequence by(Quantization quantization) {
			return new Sequence(calculate(quantization));
		}

		private String[] calculate(Quantization quantization) {
			return timeline.stream()
					.map(quantization::get)
					.toArray(String[]::new);
		}

	}

	@Override
	public String toString() {
		return Arrays.toString(tokens);
	}

	public interface Quantization {
		String get(Timeline.Point point);

		static Quantization of(Period period) {
			return point -> period.labelOf(point.instant());
		}

	}
}
