package io.intino.sumus.chronos.itl;

import io.intino.sumus.chronos.Period;
import io.intino.sumus.chronos.Timeline;

import java.time.Instant;
import java.util.function.Consumer;

import static java.lang.Double.NaN;

public class ItlParser implements Consumer<String> {
	private final Timeline.Builder builder;
	private String[] measurements;
	private Period period;
	private Instant instant;

	public ItlParser(int size) {
		builder = new Timeline.Builder(size);
	}

	@Override
	public void accept(String line) {
		if (line.startsWith("@"))
			loadAnnotation(line);
		else
			loadMeasurement(line.split("\t"));
	}

	private void loadAnnotation(String line) {
		int index = line.indexOf(' ');
		if (index < 0) return;
		String metadata = line.substring(1, index).toLowerCase();
		String data = line.substring(index + 1);
		switch (metadata) {
			case "instant":
				instant = Instant.parse(data);
				break;
			case "period":
				period = Period.each(data);
				break;
			case "measurements":
				measurements = data.split(",");
				break;
		}
	}

	private void loadMeasurement(String[] line) {
		checkModel();
		builder.set(instant);
		int size = Math.min(line.length, measurements.length);
		for (int i = 0; i < size; i++)
			builder.set(measurements[i], parse(line[i]));
		instant = period.next(instant);
	}

	private double parse(String s) {
		return s.isEmpty() ? NaN : Double.parseDouble(s);
	}

	private void checkModel() {
		if (period == null) throw new IllegalArgumentException("No period defined");
		if (instant == null) throw new IllegalArgumentException("No instant defined");
		if (measurements == null) throw new IllegalArgumentException("No measurements defined");
	}

	public Timeline timeline() {
		return builder.close();
	}
}
