package io.intino.sumus.chronos;

import io.intino.sumus.chronos.itl.ItlReader;
import io.intino.sumus.chronos.processors.Interpolator;
import io.intino.sumus.chronos.processors.Resampler;

import java.io.File;
import java.io.IOException;
import java.time.Instant;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Stream;

import static io.intino.sumus.chronos.Magnitude.NaN;
import static java.lang.Double.NaN;
import static java.lang.Math.*;
import static java.util.Arrays.binarySearch;
import static java.util.Arrays.copyOfRange;
import static java.util.stream.Collectors.*;
import static java.util.stream.IntStream.iterate;
import static java.util.stream.IntStream.range;

public class Timeline implements Iterable<Timeline.Point> {
	public static final Timeline Null = new Timeline(new Instant[0], new HashMap<>());
	public final Instant[] instants;
	private final Map<Magnitude, double[]> magnitudes;

	private Timeline(Instant[] instants, Map<Magnitude, double[]> magnitudes) {
		this.instants = instants;
		this.magnitudes = magnitudes;
	}

	public int size() {
		return magnitudes.size();
	}

	public Set<Magnitude> magnitudes() {
		return magnitudes.keySet();
	}

	public boolean has(String magnitude) {
		return has(new Magnitude(magnitude));
	}

	public boolean has(Magnitude magnitude) {
		return magnitudes.containsKey(magnitude);
	}

	public TimeSeries get(String magnitude) {
		return has(magnitude) ? get(find(magnitude)) : emptyTimeSeries();
	}

	public TimeSeries get(Magnitude magnitude) {
		return new TimeSeries(magnitude.model, instants, magnitudes.getOrDefault(magnitude, NaN(instants.length)));
	}

	private TimeSeries emptyTimeSeries() {
		return new TimeSeries(Magnitude.Model.Default, instants, NaN(instants.length));
	}

	public static Timeline empty() {
		return new Timeline(new Instant[0], new HashMap<>());
	}

	public boolean isEmpty() {
		return length() == 0;
	}

	public int length() {
		return instants.length;
	}

	public boolean isBefore(Timeline timeline) {
		if (timeline.length() == 0) return false;
		return isEndOfTimes(timeline.firstInstant());
	}

	public boolean isAfter(Timeline timeline) {
		if (timeline.length() == 0) return false;
		return isBeginningOfTimes(timeline.lastInstant());
	}

	public Point first() {
		return point(0);
	}

	public Point last() {
		return point(lastIndex());
	}

	public Point at(Instant instant) {
		return point(indexOf(instant));
	}

	public Timeline head(int length) {
		return sub(0, limitHigh(length));
	}

	public Timeline tail(int length) {
		return sub(limitLow(length() - length), length());
	}

	public Timeline from(Instant instant) {
		if (isEndOfTimes(instant)) return empty();
		return sub(indexOf(instant), length());
	}

	public Timeline from(Instant instant, int length) {
		if (isEndOfTimes(instant)) return empty();
		int from = indexOf(instant);
		return length > 0 ?
				sub(from, limitHigh(from + length)) :
				sub(limitLow(from + 1 + length), limitHigh(from + 1));
	}

	public Timeline from(Instant instant, Instant to) {
		if (isEndOfTimes(instant)) return empty();
		if (isEndOfTimes(to)) return from(instant);
		return sub(indexOf(instant), indexOf(to));
	}

	public Timeline to(Instant instant) {
		if (isEndOfTimes(instant)) return this;
		return sub(0, indexOf(instant));
	}

	private Point point(int index) {
		return isInRange(index) ? new Point(index) : null;
	}

	private boolean isBeginningOfTimes(Instant instant) {
		return firstInstant().isAfter(instant);
	}

	private boolean isEndOfTimes(Instant instant) {
		return lastInstant().isBefore(instant);
	}

	private int limitLow(int offset) {
		return max(0, offset);
	}

	private int limitHigh(int offset) {
		return min(offset, length());
	}

	private Instant firstInstant() {
		return instants[0];
	}

	private Instant lastInstant() {
		return instants[lastIndex()];
	}

	private Timeline sub(int from, int to) {
		return new Timeline(instants(from, to), values(from, to));
	}

	private Magnitude find(String label) {
		return magnitudes.keySet().stream()
				.filter(m -> m.label.equalsIgnoreCase(label))
				.findFirst()
				.orElse(null);
	}

	public Stream<Point> stream() {
		return iterate(0, i -> i < length(), i -> i + 1).mapToObj(this::point);
	}

	@Override
	public Iterator<Point> iterator() {
		return new Iterator<>() {
			int i = 0;

			@Override
			public boolean hasNext() {
				return i < length();
			}

			@Override
			public Point next() {
				return new Point(i++);
			}
		};
	}

	public Timeline add(Timeline timeline) {
		assert Arrays.equals(timeline.instants, instants);
		Timeline result = new Timeline(instants, new HashMap<>(magnitudes));
		for (Magnitude magnitude : timeline.magnitudes()) {
			TimeSeries series = result.has(magnitude) ? timeline.get(magnitude).plus(get(magnitude)) : timeline.get(magnitude);
			result.put(magnitude, series);
		}
		return result;
	}

	public Timeline add(String magnitude, TimeSeries timeSeries) {
		return add(new Magnitude(magnitude), timeSeries);
	}

	public Timeline add(Magnitude magnitude, TimeSeries timeSeries) {
		assert Arrays.equals(timeSeries.instants, instants);
		Timeline result = new Timeline(instants, new HashMap<>(magnitudes));
		result.put(magnitude, timeSeries);
		return result;
	}

	public Timeline add(String magnitude, Function<Timeline, TimeSeries> function) {
		return add(new Magnitude(magnitude), function);
	}

	public Timeline add(Magnitude magnitude, Function<Timeline, TimeSeries> function) {
		Timeline result = new Timeline(instants, new HashMap<>(magnitudes));
		result.put(magnitude, function.apply(this));
		return result;
	}

	public Timeline concat(Timeline timeline) {
		assert isBefore(timeline);
		Timeline result = new Timeline(TimeSeries.concat(instants, timeline.instants), new HashMap<>());
		Set<Magnitude> magnitudes = concat(magnitudes(), timeline.magnitudes());
		for (Magnitude magnitude : magnitudes)
			result.put(magnitude, TimeSeries.concat(get(magnitude).values, timeline.get(magnitude).values));
		return result;
	}

	public Timeline compose(Function<Magnitude, Magnitude> function) {
		Timeline result = new Timeline(instants, new HashMap<>());
		for (Magnitude magnitude : magnitudes()) {
			Magnitude m = function.apply(magnitude);
			TimeSeries series = result.has(m) ? result.get(m).plus(get(magnitude)) : get(magnitude);
			result.put(m, series);
		}
		return result;
	}

	private void put(Magnitude magnitude, TimeSeries series) {
		magnitudes.put(magnitude, series.values);
	}

	private void put(Magnitude magnitude, double[] values) {
		put(magnitude, new TimeSeries(Magnitude.Model.Default, instants, values));
	}

	public Timeline resampleBy(Period period) {
		return length() > 0 ? resampleBy(period, sizeWith(period)) : this;
	}

	public Timeline resampleBy(Period period, int size) {
		return new Resampler(this).execute(period, size);
	}

	public Timeline execute(Program program) {
		return program.run(this);
	}

	public Timeline interpolate() {
		return new Interpolator(this).execute();
	}

	private static Set<Magnitude> concat(Set<Magnitude> a, Set<Magnitude> b) {
		Set<Magnitude> magnitudes = new HashSet<>();
		magnitudes.addAll(a);
		magnitudes.addAll(b);
		return magnitudes;
	}

	private int sizeWith(Period period) {
		return period.length(first().instant(), last().instant().plusSeconds(1));
	}

	private Timeline standardize(Timeline timeline) {
		Timeline result = new Timeline(timeline.instants, new HashMap<>());
		for (Magnitude magnitude : magnitudes())
			result.put(magnitude, get(magnitude).standardize());
		return result;
	}

	@Override
	public boolean equals(Object o) {
		if (o == null || getClass() != o.getClass()) return false;
		return this == o || equals((Timeline) o);
	}

	private boolean equals(Timeline timeline) {
		return Arrays.equals(instants, timeline.instants) &&
				magnitudes.size() == timeline.size() &&
				magnitudes.keySet().stream().map(m -> Arrays.equals(magnitudes.get(m), timeline.magnitudes.get(m))).reduce(true, (a, b) -> a && b);
	}

	@Override
	public int hashCode() {
		int result = Objects.hash(magnitudes);
		result = 31 * result + Arrays.hashCode(instants);
		return result;
	}

	private int indexOf(Instant instant) {
		int index = binarySearch(instants, instant);
		return index >= 0 ? index : min(abs(-index - 1), lastIndex());
	}

	private int lastIndex() {
		return length() - 1;
	}

	private boolean isInRange(int index) {
		return index >= 0 && index < length();
	}

	private Instant[] instants(int from, int to) {
		return copyOfRange(this.instants, from, to);
	}

	private Map<Magnitude, double[]> values(int from, int to) {
		return magnitudes().stream()
				.collect(toMap(m -> m, m -> copyOfRange(valuesOf(m), from, to)));
	}

	private double[] valuesOf(Magnitude magnitude) {
		return magnitudes.get(magnitude);
	}

	public Map<String, double[]> collect(String... magnitudes) {
		return Arrays.stream(magnitudes)
				.collect(toMap(m -> m, m -> get(m).values, (a, b) -> b));
	}


	public class Point {
		final int index;

		public Point(int index) {
			this.index = index;
		}

		public Instant instant() {
			return instants[index];
		}

		public Set<Magnitude> magnitudes() {
			return Timeline.this.magnitudes();
		}

		public boolean has(String magnitude) {
			return has(new Magnitude(magnitude));
		}

		public boolean has(Magnitude magnitude) {
			return Timeline.this.has(magnitude);
		}

		public double value(String magnitude) {
			return value(new Magnitude(magnitude));
		}

		public double value(Magnitude magnitude) {
			return has(magnitude) ? valuesOf(magnitude)[index] : NaN;
		}

		@Override
		public String toString() {
			return instant() + magnitudes().stream().map(m -> "\t" + format(value(m))).collect(joining());
		}

		public Stream<Point> forward() {
			return range(index,length()).mapToObj(Timeline.this::point);
		}

		public Stream<Point> backward() {
			return iterate(index, i -> i >= 0, i -> i - 1).mapToObj(Timeline.this::point);
		}

		private String format(double value) {
			long v = (long) value;
			return value == v ? String.valueOf(v) : String.valueOf(value);
		}

		public Point next() {
			return step(1);
		}

		public Point prev() {
			return step(-1);
		}

		public Point step(int value) {
			return point(index + value);
		}

		@Override
		public boolean equals(Object o) {
			if (o == null || getClass() != o.getClass()) return false;
			return this == o || index == ((Point) o).index;
		}

		@Override
		public int hashCode() {
			return Objects.hash(index);
		}
	}

	public static class Builder {
		public final Instant[] instants;
		public final Map<Magnitude, double[]> magnitudes;
		protected int position;

		public Builder(int size) {
			this(new Instant[size]);
		}

		public Builder(Instant[] instants) {
			this.instants = instants;
			this.magnitudes = new HashMap<>();
			this.position = -1;
		}

		public Magnitude get(String magnitude) {
			return magnitudes().stream()
					.filter(m -> m.label.equals(magnitude))
					.findFirst()
					.orElse(null);
		}

		private Set<Magnitude> magnitudes() {
			return magnitudes.keySet();
		}

		public Builder put(Timeline timeline) {
			this.magnitudes.putAll(timeline.magnitudes);
			return this;
		}

		public Builder put(Magnitude magnitude, TimeSeries series) {
			assert instants.length == series.length();
			this.magnitudes.put(magnitude, series.values);
			return this;
		}

		public Builder put(Magnitude magnitude, double[] values) {
			assert instants.length == values.length;
			this.magnitudes.put(magnitude, values);
			return this;
		}

		public Builder put(Map<Magnitude, double[]> values) {
			values.keySet().forEach(m -> put(m, new TimeSeries(Magnitude.Model.Default, instants, values.get(m))));
			return this;
		}

		public Builder set(Instant instant) {
			instants[++position] = instant;
			return this;
		}

		public boolean isComplete() {
			return position + 1 >= instants.length;
		}

		public Builder set(String magnitude, double value) {
			return set(new Magnitude(magnitude), value);
		}

		public Builder set(Magnitude magnitude, double value) {
			if (magnitude == null || Double.isNaN(value)) return this;
			putIfNotExists(magnitude);
			magnitudes.get(magnitude)[position] = value;
			return this;
		}

		private void putIfNotExists(Magnitude magnitude) {
			if (magnitudes.containsKey(magnitude)) return;
			magnitudes.put(magnitude, NaN(instants.length));
		}

		public Timeline close() {
			return new Timeline(instants, magnitudes);
		}

		public TimeSeries series(Magnitude magnitude) {
			return new TimeSeries(Magnitude.Model.Default, instants, magnitudes.get(magnitude));
		}
	}

	public static Timeline read(File file) throws IOException {
		return ItlReader.read(file);
	}

	public static Timeline read(List<String> lines) {
		return ItlReader.read(lines);
	}

	public static Timeline read(String[] lines) {
		return ItlReader.read(Arrays.stream(lines).collect(toList()));
	}


}
