package io.intino.sumus.chronos;

import io.intino.sumus.chronos.Reel.Shot;
import io.intino.sumus.chronos.Reel.State;

import java.io.*;
import java.nio.file.Files;
import java.time.Instant;
import java.util.HashSet;
import java.util.Set;

import static io.intino.sumus.chronos.Reel.State.On;
import static java.lang.Long.parseLong;
import static java.lang.String.format;
import static java.nio.file.StandardOpenOption.APPEND;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class ReelFile {
	private final File file;
	private final Header header;

	private ReelFile(File file) throws IOException {
		this.file = file;
		this.header = new Header();
	}

	public static ReelFile create(File file) throws IOException {
		return new ReelFile(file);
	}

	public static ReelFile open(File file) throws IOException {
		return new ReelFile(file);
	}

	public ReelFile append(Shot... shots) throws IOException {
		boolean dirty = false;
		for (Shot shot : shots) {
			if (stateOf(shot.signal()) == shot.state()) continue;
			Files.write(file.toPath(), serialize(shot), APPEND);
			updateHeaderWith(shot);
			dirty = true;
		}
		if (dirty) header.write();
		return this;
	}

	public State stateOf(String signal) {
		return header.get(signal.hashCode());
	}

	private void updateHeaderWith(Shot shot) {
		if (shot.state() == On)
			header.put(shot.signal());
		else
			header.remove(shot.signal());
	}

	private byte[] serialize(Shot shot) {
		return format("%d\t%s\t%d\n", shot.ts().getEpochSecond(), shot.signal(), shot.state() == On ? 1 : 0)
				.getBytes();
	}

	public Processor reel(Instant from, Instant to) {
		return new Processor(from, to);
	}


	private class Header {
		private static final int SIZE = 2048;
		private final Set<Integer> hashes;

		public Header() throws IOException {
			this.hashes = file.exists() ? read() : create();
		}

		private Set<Integer> read() {
			Set<Integer> set = new HashSet<>();
			try (DataInputStream is = new DataInputStream(new FileInputStream(file))) {
				final int max = SIZE >> 2;
				while (set.size() < max) {
					int hash = is.readInt();
					if (hash == 0) break;
					set.add(hash);
				}
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
			return set;
		}

		private Set<Integer> create() throws IOException {
			try (RandomAccessFile raf = new RandomAccessFile(file, "rw")) {
				raf.write(new byte[SIZE]);
			}
			return new HashSet<>();
		}

		public void write() throws IOException {
			try (RandomAccessFile raf = new RandomAccessFile(file, "rw")) {
				raf.seek(0);
				for (int hash : hashes) raf.writeInt(hash);
				raf.write(0);
			}
		}

		public State get(String id) {
			return get(id.hashCode());
		}

		public State get(int hash) {
			return State.of(hashes.contains(hash));
		}

		public void put(String id) {
			put(id.hashCode());
		}

		public void put(int hash) {
			if (hashes.contains(hash)) return;
			hashes.add(hash);
		}

		public void remove(String id) {
			remove(id.hashCode());
		}

		public void remove(int hash) {
			if (!hashes.contains(hash)) return;
			hashes.remove(hash);
		}
	}

	public class Processor {
		private final Instant from;
		private final Instant to;

		public Processor(Instant from, Instant to) {
			this.from = from;
			this.to = to;
		}

		public Reel by(Period period) {
			Reel.Builder builder = new Reel.Builder(from, to).by(period);
			try (BufferedReader reader = new BufferedReader(new FileReader(file))) {
				reader.skip(Header.SIZE);
				reader.lines().map(this::parse).forEach(builder::add);
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
			return builder.close();
		}

		private Shot parse(String line) {
			return parse(line.split("\t"));
		}

		private Shot parse(String[] split) {
			return new Shot() {
				@Override
				public Instant ts() {
					return Instant.ofEpochSecond(parseLong(split[0]));
				}

				@Override
				public String signal() {
					return split[1];
				}

				@Override
				public State state() {
					return State.of(split[2]);
				}
			};
		}


	}
}
