package io.intino.sumus.chronos;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.IntStream;

import static java.lang.Integer.toBinaryString;
import static java.util.stream.Collectors.toMap;

public class Reel {
	public final Instant from;
	public final Instant to;
	public final int step;
	private final int length;
	private final Map<String, byte[]> signals;

	private Reel(Instant from, Instant to, int step, int length, Map<String, byte[]> signals) {
		this.from = from;
		this.to = to;
		this.step = step;
		this.length = length;
		this.signals = signals;
	}

	public Set<String> signals() {
		return signals.keySet();
	}

	public String get(String signal) {
		return get(signals.getOrDefault(signal, new byte[0]));
	}

	private String get(byte[] states) {
		StringBuilder sb = new StringBuilder(length);
		for (byte state : states) sb.append(chunk(state));
		while (sb.length() < length) sb.append(' ');
		return sb.substring(0, length);
	}

	private static final Map<Integer, String> Chunks = createChunks();

	private static String chunk(int i) {
		return Chunks.get(i & 0xFF);
	}

	private static Map<Integer, String> createChunks() {
		return IntStream.range(0, 256).boxed()
				.collect(toMap(i -> i, Reel::createChunk));
	}

	private static String createChunk(int i) {
		return toBinaryString((i & 0xFF) + 0x100)
				.substring(1)
				.replace('0', ' ')
				.replace('1', '-');
	}

	public static class Builder {
		private final Instant from;
		private final Instant to;
		private final Map<String, byte[]> signals;
		private int step;

		public Builder(Instant from, Instant to) {
			this.from = from;
			this.to = to;
			this.step = 1;
			this.signals = new HashMap<>();
		}

		public Builder add(Shot shot) {
			return shot.state() == State.On ?
					on(shot.signal(), shot.ts()) :
					off(shot.signal(), shot.ts());
		}

		public Builder on(String signal, Instant ts) {
			if (isInRange(ts))
				fillFrom(get(signal), indexOf(ts), true);
			return this;
		}

		public Builder off(String signal, Instant ts) {
			if (isInRange(ts))
				fillFrom(get(signal), indexOf(ts), false);
			return this;
		}

		private boolean isInRange(Instant ts) {
			return from.getEpochSecond() <= ts.getEpochSecond() && ts.getEpochSecond() <= to.getEpochSecond();
		}

		private byte[] get(String signal) {
			createIfNotExists(signal);
			return signals.get(signal);
		}

		private void createIfNotExists(String signal) {
			if (signals.containsKey(signal)) return;
			signals.put(signal, new byte[size()]);
		}

		private int size() {
			return (int) Math.ceil(length() / 8.);
		}

		private int length() {
			return (int) indexOf(to);
		}

		private void fillFrom(byte[] values, long index, boolean set) {
			int position = (int) (index >> 3);
			if (position < 0 || position >= values.length) return;
			values[position] = set ?
					(byte) (values[position] | mask(offset(index))) :
					(byte) (values[position] & ~mask(offset(index)));
			byte value = (byte) (set ? 0xFF : 0);

			for (int i = position+1, size = size(); i < size; i++) values[i] = value;
		}

		private static byte offset(long index) {
			return (byte) (index & 7);
		}

		private static byte mask(int offset) {
			return (byte) ((1 << (8 - offset)) - 1);
		}

		private long indexOf(Instant ts) {
			return (ts.getEpochSecond() - from.getEpochSecond()) / step;
		}

		public Reel close() {
			return new Reel(from, to, step, length(), signals);
		}

		public Builder by(Period period) {
			this.step = (int) period.duration();
			return this;
		}

		public Builder by(int seconds) {
			this.step = seconds;
			return this;
		}
	}


	public interface Shot {
		Instant ts();
		String signal();
		State state();
	}

	public enum State {
		On, Off;

		public static State of(String value) {
			return of(value.equals("1"));
		}

		public static State of(boolean value) {
			return value ? On : Off;
		}
	}

}
