package io.intino.sumus.chronos;

import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;

import static java.lang.Integer.parseInt;
import static java.time.DayOfWeek.MONDAY;
import static java.time.ZoneOffset.UTC;
import static java.time.temporal.ChronoUnit.*;
import static java.util.stream.LongStream.iterate;

public class Period {
	public static Period Seconds = each(1, SECONDS);
	public static Period Minutes = each(1, MINUTES);
	public static Period Hours = each(1, HOURS);
	public static Period Days = each(1, DAYS);
	public static Period Weeks = eachWeek(MONDAY);
	public static Period Months = each(1, MONTHS);
	public static Period Quarters = each(3, MONTHS);
	public static Period HalfYears = each(6, MONTHS);
	public static Period Years = each(1, YEARS);

	public final int amount;
	public final ChronoUnit unit;

	public static Period each(int amount, ChronoUnit unit) {
		return new Period(amount, unit);
	}

	public static Period each(short amount, short chronoUnit) {
		return new Period(amount,ChronoUnit.values()[chronoUnit]);
	}

	public static Period each(String data) {
		return parse(data.toUpperCase().toCharArray());
	}

	public static WeekPeriod eachWeek(DayOfWeek firstDayOfWeek) {
		return new WeekPeriod(1, firstDayOfWeek);
	}

	public Period(int amount, ChronoUnit unit) {
		this.amount = amount;
		this.unit = unit;
	}

	public int length(Instant from, Instant to) {
		return length(iterator(from), to.getEpochSecond());
	}

	private int length(Iterator<Instant> iterator, long to) {
		return (int) iterate(nextIn(iterator), next -> next < to, next -> nextIn(iterator)).count();
	}

	private static long nextIn(Iterator<Instant> iterator) {
		return iterator.next().getEpochSecond();
	}

	public Instant crop(String instant) {
		return crop(Instant.parse(instant));
	}

	public Instant crop(Instant instant) {
		switch (unit) {
			case YEARS:
				return cropYear(split(instant));
			case MONTHS:
				return cropMonth(split(instant));
			case WEEKS:
			case DAYS:
				return cropDay(split(instant));
		}
		return crop(instant.getEpochSecond());
	}

	private Instant cropYear(int[] split) {
		return toInstant(LocalDate.of(split[0], 1, 1));
	}

	private Instant cropMonth(int[] split) {
		return toInstant(LocalDate.of(split[0], cropMonth(split[1]), 1));
	}

	private int cropMonth(int month) {
		return ((month-1) - (month-1)%amount)+1;
	}

	private Instant cropWeek(int[] split) {
		return toInstant(LocalDate.of(split[0], split[1], split[2]));
	}

	private Instant cropDay(int[] split) {
		return toInstant(LocalDate.of(split[0], split[1], split[2]));
	}

	private static Instant toInstant(LocalDate date) {
		return date.atStartOfDay().atZone(UTC).toInstant();
	}

	private int[] split(Instant instant) {
		String str = instant.toString();
		return new int[] {
				parseInt(str.substring(0, 4)),
				parseInt(str.substring(5, 7)),
				parseInt(str.substring(8, 10))
		};
	}

	private Instant crop(long epochSecond) {
		return Instant.ofEpochSecond(epochSecond - epochSecond % duration());
	}

	private static Map<String, ChronoUnit> Units = units();

	private static Period parse(char[] chars) {
		int amount = amountIn(chars);
		ChronoUnit unit = Units.getOrDefault(unitIn(chars),DAYS);
		return new Period(amount, unit);
	}

	private static String unitIn(char[] chars) {
		StringBuilder unit = new StringBuilder();
		for (char c : chars) {
			if (c < 'A' || c > 'Z') continue;
			unit.append(c);
		}
		String s = unit.toString();
		return s.endsWith("S") ? singular(s) : s;
	}

	private static int amountIn(char[] chars) {
		int amount = 0;
		for (char c : chars) {
			if (c < '0' || c > '9') break;
			amount = amount * 10 + (c - '0');
		}
		return amount == 0 ? 1 : amount;
	}

	@Override
	public String toString() {
		return amount + " " + toString(unit.toString().toLowerCase());
	}

	private String toString(String unit) {
		return amount == 1 ? singular(unit) : unit;
	}

	private static String singular(String unit) {
		return unit.substring(0, unit.length() - 1);
	}

	@Override
	public boolean equals(Object o) {
		if (o == null || getClass() != o.getClass()) return false;
		return this == o || equals((Period) o);
	}

	private boolean equals(Period period) {
		return amount == period.amount && unit == period.unit;
	}

	@Override
	public int hashCode() {
		return Objects.hash(amount, unit);
	}

	public Iterator<Instant> iterator(Instant from) {
		switch (unit) {
			case WEEKS:
				return new WeekIterator(from);
			case MONTHS:
				return new MonthIterator(from);
			case YEARS:
				return new YearIterator(from);
		}
		return new DefaultIterator(from);
	}

	public String labelOf(Instant instant) {
		return trim(crop(instant).toString());
	}

	private String trim(String str) {
		return clean(str).substring(0, labelLength(duration()));
	}

	private String clean(String str) {
		StringBuilder sb = new StringBuilder();
		for (char c : str.toCharArray())
			if (c >= '0' && c <= '9') sb.append(c);
		return sb.toString();
	}

	private static final Map<Long,Integer> LabelLengths = Map.of(
			Years.duration(), 4,
			Months.duration(), 6,
			Days.duration(), 8,
			Hours.duration(), 10,
			Minutes.duration(), 12
	);

	private int labelLength(long duration) {
		return LabelLengths.keySet().stream()
				.sorted((a,b)->Long.compare(b,a))
				.filter(d -> duration >= d)
				.mapToInt(LabelLengths::get)
				.findFirst()
				.orElse(14);
	}

	private class DefaultIterator implements Iterator<Instant> {
		private final long time;
		private long next;

		private DefaultIterator(Instant instant) {
			this.time = duration();
			this.next = crop(instant).getEpochSecond();
		}

		@Override
		public boolean hasNext() {
			return true;
		}

		public Instant next() {
			Instant result = Instant.ofEpochSecond(next);
			next += time;
			return result;
		}

	}

	private class WeekIterator implements Iterator<Instant> {
		private Instant next;

		private WeekIterator(Instant instant) {
			this.next = crop(instant);
		}

		@Override
		public boolean hasNext() {
			return true;
		}

		@Override
		public Instant next() {
			Instant result = next;
			next = next.plusSeconds(duration());
			return result;
		}

	}

	private class MonthIterator implements Iterator<Instant> {
		private LocalDate next;

		private MonthIterator(Instant instant) {
			this.next = LocalDate.ofInstant(crop(instant), UTC);
		}

		@Override
		public boolean hasNext() {
			return true;
		}

		@Override
		public Instant next() {
			Instant result = next.atStartOfDay().toInstant(UTC);
			next = next.plusMonths(amount);
			return result;
		}

	}

	private class YearIterator implements Iterator<Instant> {
		private LocalDate next;

		private YearIterator(Instant instant) {
			this.next = LocalDate.ofInstant(crop(instant),UTC);
		}

		@Override
		public boolean hasNext() {
			return true;
		}

		@Override
		public Instant next() {
			Instant result = next.atStartOfDay().toInstant(UTC);
			next = next.plusYears(amount);
			return result;
		}
	}

	public Instant next(Instant instant) {
		Iterator<Instant> iterator = iterator(instant);
		iterator.next();
		return iterator.next();
	}

	public long duration() {
		return amount * unit.getDuration().toSeconds();
	}

	private static Map<String, ChronoUnit> units() {
		return Map.of(
				"SECOND", SECONDS,
				"MINUTE", MINUTES,
				"HOUR", HOURS,
				"DAY", DAYS,
				"WEEK", WEEKS,
				"MONTH", MONTHS,
				"YEAR", YEARS
		);
	}

	public static class WeekPeriod extends Period {
		private final DayOfWeek firstDayOfWeek;

		public WeekPeriod(int amount, DayOfWeek firstDayOfWeek) {
			super(amount, WEEKS);
			this.firstDayOfWeek = firstDayOfWeek;
		}

		@Override
		public Iterator<Instant> iterator(Instant from) {
			return new WeekIterator(firstDayOfWeek(from));
		}

		private Instant firstDayOfWeek(Instant instant) {
			Instant next = dailyIterator(instant).next().plus(-6, DAYS);
			while (dayOfWeek(next) != firstDayOfWeek)
				next = next.plus(1, DAYS);
			return next;
		}

		private static Iterator<Instant> dailyIterator(Instant instant) {
			return Days.iterator(instant);
		}

		private static DayOfWeek dayOfWeek(Instant instant) {
			return instant.atZone(UTC).getDayOfWeek();
		}
	}
}
