package io.intino.sumus.time.models.descriptive.timeseries;

import com.tdunning.math.stats.TDigest;
import io.intino.sumus.time.Magnitude;
import io.intino.sumus.time.TimeSeries;
import org.apache.commons.math3.distribution.NormalDistribution;
import org.apache.commons.math3.distribution.PoissonDistribution;

import static io.intino.sumus.time.Magnitude.Model.DistributionModel.Poisson;
import static io.intino.sumus.time.Magnitude.Model.DistributionModel.Unknown;
import static io.intino.sumus.time.Magnitude.Model.DistributionTail.Down;
import static io.intino.sumus.time.Magnitude.Model.DistributionTail.Up;
import static java.lang.Double.NaN;
import static java.lang.Double.isNaN;

public class Distribution {
	final Magnitude.Model model;
	private final TDigest tdigest;
	public double max = Integer.MIN_VALUE;
	public double min = Integer.MAX_VALUE;
	public double open;
	public double close;
	public double sum;
	public int count;
	private double mean;
	private double m2;

	public static Distribution of(TimeSeries timeSeries) {
		return new Distribution(timeSeries.model).add(timeSeries.values);
	}

	public Distribution(TimeSeries timeSeries) {
		this(timeSeries.model);
		for (TimeSeries.Point point : timeSeries) add(point.value());
	}

	public Distribution(Magnitude.Model model) {
		this.model = model;
		this.tdigest = model.distributionModel == Unknown ? TDigest.createAvlTreeDigest(200) : null;
	}

	Distribution add(double[] values) {
		for (double value : values) add(value);
		return this;
	}

	public void add(double value) {
		if (isNaN(value)) return;
		open = count == 0 ? value : open;
		close = value;
		min = Math.min(min, value);
		max = Math.max(max, value);
		sum += value;
		count++;

		double delta = value - mean;
		mean += delta / count;
		m2 += delta * (value - mean);
		if (tdigest != null) tdigest.add(value);
	}

	public double mean() {
		return mean;
	}
	//Welford algorithm

	public double sd() {
		return count >= 2 ? Math.sqrt(m2 / (count - 1)) : NaN;
	}

	public double probabilityOf(double value) {
		if (model.distributionModel == Poisson) return tail(poissonCumulativeProbabilityOf((int) value));
		if (model.distributionModel == Unknown) return tail(tdigest.cdf(value));
		return tail(normalCumulativeProbabilityOf(value));
	}

	public int percentile(double value) {
		return (int) (probabilityOf(value) * 100);
	}

	public int quartile(double value) {
		return (int) (probabilityOf(value) * 4);
	}


	private double tail(double p) {
		if (model.distributionTail == Down) return p;
		if (model.distributionTail == Up) return 1-p;
		return Math.min(p, 1-p) * 2;
	}

	private double poissonCumulativeProbabilityOf(int quantile) {
		return new PoissonDistribution(mean).cumulativeProbability(quantile);
	}

	private double normalCumulativeProbabilityOf(double quantile) {
		return new NormalDistribution(mean, sd()).cumulativeProbability(quantile);
	}

	@Override
	public String toString() {
		return String.format("min=%s\nmax=%s\nopen=%s\nclose=%s\nsum=%s\ncount=%d\nmean=%s\nsd=%s\n", min, max, open, close, sum, count, mean,sd());
	}

}
