package io.intino.sumus.time;

import io.intino.sumus.time.Magnitude.Model.DistributionModel;
import io.intino.sumus.time.Magnitude.Model.DistributionTail;
import io.intino.sumus.time.Magnitude.Model.Operator;

import java.util.*;
import java.util.function.Function;
import java.util.stream.DoubleStream;

import static java.lang.Double.NaN;
import static java.lang.Double.parseDouble;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.*;

public class Magnitude implements CharSequence  {
	public final String label;
	public final Model model;

	public Magnitude(String definition) {
		this(definition.split(":"));
	}

	public Magnitude(String[] definition) {
		this(definition[0], Model.of(definition));
	}

	public Magnitude(String label, Model model) {
		this.label = label;
		this.model = model;
	}

	public Magnitude dimension(int index) {
		String[] split = label.split("\\|");
		String label = index >= 0 && index < split.length ? split[index] : "Total";
		return new Magnitude(label, model);
	}

	public Magnitude level(int index) {
		if (label.contains("|")) return this;
		int position = find(index);
		return position > 0 ? new Magnitude(label.substring(0, position), model) : this;
	}

	private int find(int index) {
		int position = 0;
		while (index > 0 && position >= 0) {
			position = label.indexOf('.',position+1);
			index--;
		}
		return position;
	}

	public int length() {
		return label.length();
	}

	@Override
	public char charAt(int index) {
		return label.charAt(index);
	}

	@Override
	public CharSequence subSequence(int start, int end) {
		return label.subSequence(start,end);
	}

	public double reduce(double[] values) {
		return model.operator.reduce(values);
	}

	public double reduce(DoubleStream values) {
		return model.operator.reduce(values);
	}

	@Override
	public int hashCode() {
		return label.hashCode();
	}

	@Override
	public boolean equals(Object o) {
		return o != null && getClass() == o.getClass() && equals((Magnitude) o);
	}

	private boolean equals(Magnitude magnitude) {
		return this == magnitude || Objects.equals(label, magnitude.label);
	}

	@Override
	public String toString() {
		String model = this.model.toString();
		return model.isEmpty() ? label : label + ":" + model;
	}

	public static double[] NaN(int size) {
		double[] values = new double[size];
		Arrays.fill(values, NaN);
		return values;
	}

	public double min() {
		return model.min;
	}

	public double max() {
		return model.max;
	}

	public String unit() {
		return model.unit;
	}

	public String symbol() {
		return model.symbol;
	}

	public Operator operator() {
		return model.operator;
	}

	public DistributionModel distributionModel() {
		return model.distributionModel;
	}

	public DistributionTail distributionTail() {
		return model.distributionTail;
	}

	public String format(double value) {
		return model.format(value);
	}

	public static class Model {
		public static Model Default = new Model(Map.of());
		private final Map<String, String> attributes;
		public final double max;
		public final double min;
		public final String unit;
		public final String symbol;
		public final DistributionModel distributionModel;
		public final DistributionTail distributionTail;
		public final Format format;
		public final Operator operator;

		public static Model of(String definition) {
			return of(definition.split(":"));
		}

		public static Model of(String[] attributes) {
			Map<String,String> result = new HashMap<>();
			for (String attribute : attributes) {
				if (attribute.indexOf('=') < 0) continue;
				String[] data = attribute.split("=");
				result.put(data[0], data.length > 1 ? data[1] : "");
			}
			return new Model(result);
		}

		public Model(Map<String, String> attributes) {
			this.attributes = attributes;
			this.max = getDouble("max");
			this.min = getDouble("min");
			this.unit = attributes.getOrDefault("unit", "");
			this.symbol = attributes.getOrDefault("symbol", "");
			this.operator = Operator.of(attributes.getOrDefault("operator", "sum"));
			this.distributionModel = DistributionModel.of(attributes.get("distribution"));
			this.distributionTail = DistributionTail.of(attributes.get("tail"));
			this.format = Format.of(attributes.get("format"));
		}

		private double getDouble(String attribute) {
			return attributes.containsKey(attribute) ? parseDouble(attributes.get(attribute)) : NaN;
		}

		public Set<String> attributes() {
			return attributes.keySet();
		}

		public String attribute(String name) {
			return attributes.getOrDefault(name, "");
		}

		@Override
		public boolean equals(Object o) {
			if (o == null || getClass() != o.getClass()) return false;
			return this == o || Objects.equals(attributes, ((Model) o).attributes);
		}

		@Override
		public int hashCode() {
			return Objects.hash(attributes);
		}

		@Override
		public String toString() {
			return attributes().stream().sorted().map(this::toString).collect(joining(":"));
		}

		private String toString(String attribute) {
			return attribute + '=' + attribute(attribute);
		}

		public Model max(int value) {
			return updateWith(Map.of("max", String.valueOf(value)));
		}

		public Model min(int value) {
			return updateWith(Map.of("min", String.valueOf(value)));
		}

		public Model unit(String value) {
			return updateWith(Map.of("unit", value));
		}

		public Model symbol(String value) {
			return updateWith(Map.of("symbol", value));
		}

		public Model operator(String value) {
			return updateWith(Map.of("operator", value));
		}

		public Model distribution(String value) {
			return updateWith(Map.of("distribution", value));
		}

		public Model updateWith(String definition) {
			return Model.of(definition.split(":"));
		}

		public Model updateWith(Map<String, String> data) {
			HashMap<String, String> attributes = new HashMap<>(this.attributes);
			attributes.putAll(data);
			return new Model(attributes);
		}

		public String format(double value) {
			return format.format(value);
		}

		public enum Operator {
			Sum(Operator::sum), Average(Operator::average),
			MostFrequent(Operator::mostFrequent),
			Count(Operator::count),
			Max(Operator::max), Min(Operator::min),
			First(Operator::first), Last(Operator::last);

			private final Function<DoubleStream, Double> function;

			Operator(Function<DoubleStream, Double> function) {
				this.function = function;
			}

			public static Operator of(String type) {
				switch (type.toLowerCase()) {
					case "avg": case "average": return Average;
					case "mostfrequent": return MostFrequent;
					case "first": return First;
					case "last": return Last;
					case "max": return Max;
					case "min": return Min;
					case "count": return Count;
				}
				return Sum;
			}

			public double reduce(double[] values) {
				return function.apply(DoubleStream.of(values));
			}

			public double reduce(DoubleStream values) {
				return function.apply(values);
			}

			public static double count(DoubleStream stream) {
				return stream.count();
			}

			public static double sum(DoubleStream stream) {
				int[] count = {0};
				double sum = stream.peek(d->count[0]++).sum();
				return count[0] > 0 ? sum : NaN;
			}

			public static double average(DoubleStream stream) {
				return stream.average().orElse(NaN);
			}

			private static Double max(DoubleStream stream) {
				return stream.average().stream().max().orElse(NaN);
			}

			private static Double min(DoubleStream stream) {
				return stream.average().stream().min().orElse(NaN);
			}

			private static Double mostFrequent(DoubleStream stream) {
				return stream.boxed()
						.collect(groupingBy(identity(), counting()))
						.entrySet().stream()
						.max(Map.Entry.comparingByValue())
						.map(Map.Entry::getKey)
						.orElse(NaN);
			}

			private static Double first(DoubleStream stream) {
				return stream.average().stream().findFirst().orElse(NaN);
			}

			private static Double last(DoubleStream stream) {
				return stream.average().stream().reduce((first, second) -> second).orElse(NaN);
			}



			@Override
			public String toString() {
				return this.name();
			}
		}

		public enum DistributionModel {
			Normal, Poisson, Unknown;

			public static DistributionModel of(String value) {
				if (value != null)
					switch (value.toUpperCase()) {
						case "POISSON": return Poisson;
						case "UNKNOWN": return Unknown;
					}
				return Normal;
			}
		}

		public enum DistributionTail {
			Down, Up, Both;

			public static DistributionTail of(String value) {
				if (value != null)
					switch (value.toUpperCase()) {
						case "DOWN": return Down;
						case "UP": return Up;
					}
				return Both;
			}


		}

	}
}
