package io.intino.sumus.time;

import java.text.DecimalFormat;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

import static java.lang.Double.NaN;
import static java.lang.Double.parseDouble;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;

public interface Format {

	class Default {
		private static Format instance = new DecimalFormat("#,##0.##")::format;

		public static Format get() {
			return instance;
		}

		public static void set(Format format) {
			Default.instance = format;
		}
	}

	static Format of(String pattern) {
		return pattern == null ? Default.get() : new CategoryFormat(pattern);
	}

	String format(double value);

	class CategoryFormat implements Format {
		private final List<Function<Double, String>> functions;

		public CategoryFormat(String pattern) {
			this(pattern.split(","));
		}

		private CategoryFormat(String[] categories) {
			this.functions = stream(categories).map(this::parse).collect(toList());
		}

		private Function<Double, String> parse(String category) {
			return parse(category.split(" "));
		}

		private Function<Double, String> parse(String[] split) {
			try {
				return asNumber(split);
			} catch (NumberFormatException e) {
				return asRange(split);
			}
		}

		private Function<Double, String> asRange(String[] split) {
			Format.Range range = Format.Range.of(split[1]);
			return v -> range.matches(v) ? split[0] : null;
		}

		private static Function<Double, String> asNumber(String[] split) {
			double value = parseDouble(split[1]);
			return v -> v == value ? split[0] : null;
		}

		@Override
		public String format(double value) {
			return functions.stream()
					.map(f -> f.apply(value))
					.filter(Objects::nonNull)
					.findFirst()
					.orElse(Default.get().format(value));
		}

	}

	class Range {
		private final double low;
		private final double high;
		private final boolean lowClosed;
		private final boolean highClosed;

		public Range(double[] values, boolean lowClosed, boolean highClosed) {
			this.low = values[0];
			this.high = values[1];
			this.lowClosed = lowClosed;
			this.highClosed = highClosed;
		}

		public static Range of(String definition) {
			return new Range(values(definition), isLowClosed(definition), isHighClosed(definition));
		}

		private static boolean isHighClosed(String definition) {
			return definition.endsWith("]");
		}

		private static boolean isLowClosed(String definition) {
			return definition.startsWith("[");
		}

		private static double[] values(String definition) {
			try {
				return stream(split(definition)).mapToDouble(Double::parseDouble).toArray();
			} catch (NumberFormatException e) {
				return new double[]{NaN, NaN};
			}
		}

		private static String[] split(String definition) {
			return definition.replaceAll("[\\[\\]\\(\\)]", "").split("\\.\\.");
		}

		public boolean matches(double value) {
			return matchesLow(value) && matchesHigh(value);
		}

		private boolean matchesHigh(double value) {
			return highClosed ? value <= high : value < high;
		}

		private boolean matchesLow(double value) {
			return lowClosed ? value >= low : value > low;
		}
	}
}
