package io.intino.sumus.box;

import io.intino.konos.alexandria.Box;
import io.intino.konos.alexandria.ui.model.Catalog;
import io.intino.konos.alexandria.ui.model.Item;
import io.intino.konos.alexandria.ui.model.TimeRange;
import io.intino.konos.alexandria.ui.model.catalog.Scope;
import io.intino.konos.alexandria.ui.model.catalog.arrangement.Group;
import io.intino.konos.alexandria.ui.services.push.UISession;
import io.intino.sumus.QueryEngine;
import io.intino.sumus.graph.*;
import io.intino.sumus.helpers.NameSpaceHandler;
import io.intino.sumus.queries.EntityQuery;
import io.intino.sumus.queries.Filter;
import io.intino.sumus.queries.TemporalRecordQuery;
import io.intino.tara.magritte.Concept;
import io.intino.tara.magritte.Layer;

import java.util.*;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

public class SumusDisplayHelper {
	private static Map<String, QueryEngine> queryEngineMap = new HashMap<>();
	private static NameSpaceHandler nameSpaceHandler;

	public static QueryEngine queryEngine(SumusBox box, UISession session) {
		String id = session.id();
		if (!queryEngineMap.containsKey(id))
			queryEngineMap.put(id, new QueryEngine(box));
		return queryEngineMap.get(id);
	}

	public static SumusBox sumusBox(Box box) {
		return (SumusBox) box.owner();
	}

	public static NameSpaceHandler nameSpaceHandler(SumusBox box) {
		if (nameSpaceHandler == null)
			nameSpaceHandler = new NameSpaceHandler(box.graph());
		return nameSpaceHandler;
	}

	public static io.intino.sumus.queries.Scope scopeOf(Scope scope) {
		if (scope == null) return null;
		return new io.intino.sumus.queries.Scope().tags(tags(scope.groups())).records(records(scope.objects()));
	}

	public static List<AbstractAccess> accesses(SumusBox box, UISession session) {
		return box.graph().core$().as(SumusGraph.class).accesses(session.user());
	}

	public static Catalog.ArrangementFilterer createArrangementFilterer(SumusBox box, UISession session) {
		return new Catalog.ArrangementFilterer() {
			public List<String> groupings = new ArrayList<>();
			Filter filter = null;

			@Override
			public UISession session() {
				return session;
			}

			@Override
			public void add(String grouping, Group... groups) {
				groupings.add(grouping);
				filter = isEmpty() ? new Filter() : filter;
				List<String> tags = tags(groups);
				filter.addTags(tags);
			}

			@Override
			public boolean contains(Item item) {
				if (item == null) return false;
				return filter.contains(groupings.stream()
						.map(g -> categorization(g).categorizedLayersMap(Collections.singletonList(item.object())).keySet())
						.flatMap(Collection::stream)
						.collect(toList()));
			}

			private Categorization categorization(String grouping) {
				return box.graph().categorizationList(c -> c.name$().equals(grouping)).collect(toList()).get(0);
			}

			@Override
			public void clear() {
				filter = null;
				groupings.clear();
			}

			@Override
			public boolean isEmpty() {
				return filter == null;
			}
		};
	}

	private static Map<String, List<String>> tags(Map<String, List<Group>> groups) {
		return groups.entrySet().stream().collect(toMap(Map.Entry::getKey, e -> tags(e.getValue())));
	}

	public static List<String> tags(Group... groups) {
		return tags(Stream.of(groups));
	}

	public static List<String> tags(List<Group> groups) {
		return tags(groups.stream());
	}

	public static String tag(Group group) {
		return group.label();
	}

	public static List<Group> groups(SumusBox box, Categorization categorization, List<? extends Layer> items, UISession session) {
		return groups(categorization.filter(categorization.categorizedLayersMap(items), new LinkedHashSet<>(categorization.tags(accesses(box, session)))));
	}

	public static Group group(String tag, int count) {
		return new Group().label(tag).countObjects(count);
	}

	public static void createGroup(SumusBox box, Catalog catalog, String grouping, Group group, String username) {
		// TODO MC -> filtros
		//ClusterHelper.registerClusterGroup(box, catalog, grouping, group.label(), objectsToEntities(group.objects()), username);
	}

	private static List<String> tags(Stream<Group> groups) {
		return groups.map(SumusDisplayHelper::tag).collect(toList());
	}

	private static Map<String, List<Record>> records(Map<String, List<Object>> objects) {
		return objects.entrySet().stream().collect(toMap(Map.Entry::getKey, e -> objectsToRecords(e.getValue())));
	}

	private static List<String> objectsToRecordsIds(List<Object> objects) {
		return objects.stream().map(o -> ((Layer) o).core$().id()).collect(toList());
	}

	private static List<Record> objectsToRecords(List<Object> objects) {
		return objects.stream().map(o -> ((Layer) o).core$().as(Record.class)).collect(toList());
	}

	private static List<Entity> objectsToEntities(List<Object> objects) {
		return objects.stream().map(o -> ((Layer) o).core$().as(Entity.class)).collect(toList());
	}

	private static List<Record> layersToRecords(List<? extends Layer> objects) {
		return objects.stream().map(o -> o.a$(Record.class)).collect(toList());
	}

	public static <T extends Layer> List<T> entities(SumusBox box, Class<T> entityClass, Scope scope, String condition, UISession session) {
		Concept concept = box.graph().core$().concept(entityClass);
		EntityQuery.Builder builder = new EntityQuery.Builder();
		builder.scope(scopeOf(scope));
		builder.condition(condition);
		return queryEngine(box, session).entities(builder.build(concept, session.user())).stream().map(e -> e.a$(entityClass)).collect(toList());
	}

	public static <T extends Layer> T entity(SumusBox box, Class<T> entityClass, String id, UISession session) {
		Entity entity = queryEngine(box, session).entity(id, session.user());
		return entity != null ? entity.a$(entityClass) : null;
	}

	public static <T extends Layer> List<T> temporalRecords(SumusBox box, Class<T> temporalRecordClass, Scope scope, String condition, TimeRange timeRange, NameSpace nameSpace, UISession session) {
		Concept concept = box.graph().core$().concept(temporalRecordClass);
		TemporalRecordQuery.Builder builder = new TemporalRecordQuery.Builder();
		builder.scope(scopeOf(scope));
		builder.condition(condition);
		builder.timeRange(timeRange);
		builder.nameSpace(nameSpace);
		return queryEngine(box, session).temporalRecords(builder.build(concept, session.user())).stream().map(r -> r.a$(temporalRecordClass)).collect(toList());
	}

	public static <T extends Layer> T temporalRecord(SumusBox box, Class<T> temporalRecordClass, String id, UISession session) {
		TemporalRecord temporalRecord = queryEngine(box, session).temporalRecord(id, session.user());
		return temporalRecord != null ? temporalRecord.a$(temporalRecordClass) : null;
	}

	private static List<Group> groups(TagMap tagMap) {
		return tagMap.entrySet().stream().map(e -> group(e.getKey(), e.getValue().size())).collect(toList());
	}
}