package io.intino.sumus.analytics.categorization;

import io.intino.konos.alexandria.ui.model.TimeRange;
import io.intino.konos.alexandria.ui.model.TimeScale;
import io.intino.sumus.graph.AbstractAccess;
import io.intino.sumus.graph.Categorization;
import io.intino.sumus.helpers.TranslatorHelper;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;

import static java.util.stream.Collectors.toList;

public class TemporalCategorization extends Categorization {
	protected String name;
	protected String format;
	protected TranslatorHelper helper;
	protected String language;
	protected TimeRange range;
	protected List<Tag> tags = new ArrayList<>();

	public TemporalCategorization(String name, String format, Categorization parent, TimeRange range) {
		super(null);
		this.name = name;
		this.parent = parent;
		this.format = format;
		update(range);
		createTags();
	}

	public TemporalCategorization(TimeRange range) {
		this("TemporalCategorization", "", null, range);
	}

	public TemporalCategorization helper(TranslatorHelper helper) {
		this.helper = helper;
		setup();
		return this;
	}

	@Override
	public Categorization parent() {
		return parent;
	}

	private void update(TimeRange timeRange) {
		TimeScale scale = timeRange.scale();
		this.range = new TimeRange(scale.normalise(timeRange.from()), scale.normalise(timeRange.to()), scale);
		Optional.ofNullable(parent()).ifPresent(parent -> ((TemporalCategorization) parent).update(timeRange));
	}

	public TemporalCategorization language(String language) {
		this.language = language;
		setup();
		return this;
	}

	@Override
	public String name$() {
		return this.name;
	}

	public TimeRange range() {
		return range;
	}

	public String format() {
		return format;
	}

	public boolean isAvailableForScale(TimeScale scale) {
		return scale.ordinal() >= this.range().scale().ordinal();
	}

	public void range(TimeRange timeRange) {
		update(timeRange);
	}

	private void setup() {
		if (this.helper == null || this.language == null) return;
		this.label(helper.translate(name).into(language));
	}

	private void createTags() {
		range.allInstants().forEach(instant -> tags.add(new Tag(instant, range.scale().sortingWeight(instant, format))));
	}

	public List<String> tags(List<AbstractAccess> accessList) {
		List<Tag> tags = new ArrayList<>(this.tags);
		tags.sort(Comparator.comparingInt(c -> c.weight));
		return tags.stream()
				.map(t -> labelOf(t.instant))
				.collect(toList());
	}

	private String labelOf(Instant instant) {
		return range.scale().toString(instant);
	}

	private class Tag {
		Instant instant;
		int weight;

		Tag(Instant instant, int weight) {
			this.instant = instant;
			this.weight = weight;
		}
	}
}
